<?php

namespace Drupal\paragraph_group\Paragroup;

use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Service for paragraph_group helper functions.
 */
class ParagroupHelperService {

  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  public function __construct(
    ConfigFactory $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    RendererInterface $renderer,
  ) {

    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->renderer = $renderer;

  }

  /**
   * Helper function returning Drupal's main version number.
   *
   * @return string
   *   The major version number.
   */
  public function getDrupalVersion(): string {

    $version = \Drupal::VERSION;
    $version = explode('.', $version);

    return $version[0];

  }

  /**
   * Validates if the chosen administration theme is permissible.
   *
   * @param string $admin_theme
   *   The admin theme to validate.
   *
   * @return bool
   *   TRUE if the theme is valid, FALSE otherwise.
   */
  public function validateAdminTheme(string $admin_theme): bool {

    $compatible_themes = NULL;
    $version = $this->getDrupalVersion();

    if ($version == 10) {
      $compatible_themes = ['claro', 'gin'];
    }
    elseif ($version == 11) {
      $compatible_themes = ['claro', 'gin'];
    }

    if (
      $compatible_themes === NULL ||
      !in_array($admin_theme, $compatible_themes)
    ) {
      return FALSE;
    }

    return TRUE;

  }

  /**
   * Gets theme modification boolean settings.
   *
   * @return array<string, bool|string>
   *   Array of theme modification settings with keys 'sc', 'fwf', 'ver'.
   */
  private function getThemeModBoolSettings(): array {

    $config = $this->configFactory->get('paragraph_group.settings');
    $theme_mods_config = $config->get('paragraph_group.theme_mods_boxes');
    $version = $this->getDrupalVersion();

    is_array($theme_mods_config) && isset($theme_mods_config['sidebar_config']) ?
      $sc = boolval($theme_mods_config['sidebar_config']) :
      $sc = FALSE;

    is_array($theme_mods_config) && isset($theme_mods_config['full_width_forms']) ?
      $fwf = boolval($theme_mods_config['full_width_forms']) :
      $fwf = FALSE;

    return [
      'sc' => $sc,
      'fwf' => $fwf,
      'ver' => $version,
    ];

  }

  /**
   * Adds the theme modification config to the attached drupalSettings.
   *
   * @param array<string, mixed> $attached
   *   The attached array to modify.
   *
   * @return bool
   *   TRUE if successful.
   */
  public function addThemeModConfig(array &$attached): bool {

    $theme_mods_bool = $this->getThemeModBoolSettings();

    foreach ($theme_mods_bool as $key => $val) {

      if ($val) {

        if (
          !isset($attached['drupalSettings']) ||
          !is_array($attached['drupalSettings'])
        ) {
          $attached['drupalSettings'] = [];
        }

        $attached['drupalSettings']['paragraph_group_' . $key] = $val;

      }

    }

    return TRUE;

  }

  /**
   * Adds a value to an attached array key, ensuring it exists first.
   *
   * @param array<string, mixed> $attached
   *   The attached array to modify.
   * @param string $key
   *   The key to add to.
   * @param mixed $value
   *   The value to add (appended to array or merged if array).
   */
  private function addToAttachedKey(array &$attached, string $key, mixed $value): void {

    if (!isset($attached[$key]) || !is_array($attached[$key])) {
      $attached[$key] = [];
    }

    /** @var array<int|string, mixed> $target */
    $target = $attached[$key];

    if (is_array($value)) {
      $target = array_merge($target, $value);
    }
    else {
      $target[] = $value;
    }

    $attached[$key] = $target;

  }

  /**
   * Attaches the details widget library to the form attached array.
   *
   * @param array<string, mixed> $attached
   *   The attached array to modify.
   */
  public function attachDetailsWidget(array &$attached): void {

    $config_data = [
      'library' => 'paragraph_group/main',
      'drupalSettings' => ['paragraph_group_details_widget' => TRUE],
    ];

    foreach ($config_data as $key => $val) {
      $this->addToAttachedKey($attached, $key, $val);
    }

    // Attach theme modification config info as additional js settings.
    $this->addThemeModConfig($attached);

  }

  /**
   * Gets the list of bundles this module is formatting.
   *
   * @return array<string, string>|null
   *   Array of bundle machine names.
   */
  public function getFieldGroupBundles(): ?array {

    $config = $this->configFactory->get('paragraph_group.settings');
    $boxes_name = 'paragraph_group.field_groups_boxes';
    $bundles = $config->get($boxes_name);

    if (is_array($bundles) && !empty($bundles)) {

      foreach ($bundles as $key => $val) {
        if (!$val) {
          unset($bundles[$key]);
        }
      }

      // Ensure all values are strings.
      /** @var array<string, string> $filtered_bundles */
      $filtered_bundles = array_filter($bundles, 'is_string');

      return $filtered_bundles;

    }

    return NULL;

  }

  /**
   * Gets the human readable name of a Content Type from its machine name.
   *
   * @param string $bundle
   *   The machine name of the bundle.
   *
   * @return string|null
   *   The human readable name, or null if content type not found.
   */
  public function getContentTypeName(string $bundle): ?string {

    $type = $this->entityTypeManager
      ->getStorage('node_type')
      ->load($bundle);

    if ($type !== NULL) {

      $name = $type->get('name');

      is_string($name) ?
        $content_type_name = $name :
        $content_type_name = '';

      return $content_type_name;

    }

    return NULL;

  }

  /**
   * Shortcut to get the text for a link.
   *
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|string $text
   *   The link text.
   * @param string $route
   *   The route name.
   *
   * @return string
   *   The rendered link markup.
   */
  public function getLink(TranslatableMarkup|string $text, string $route): string {

    $link = Link::createFromRoute($text, $route);
    $link_render_array = $link->toRenderable();

    if (
     !isset($link_render_array['#attributes']) ||
     !is_array($link_render_array['#attributes'])
    ) {
      $link_render_array['#attributes'] = [];
    }

    $link_render_array['#attributes']['target'] = '_blank';
    $link_markup = $this->renderer->render($link_render_array);

    return (string) $link_markup;

  }

}
