<?php

namespace Drupal\paragraph_group\Plugin\Field\FieldWidget;

use Drupal\Component\Utility\Html;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\paragraphs\Plugin\Field\FieldWidget\InlineParagraphsWidget;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Paragraphs Details widget.
 *
 * @FieldWidget(
 *   id = "paragraph_group_details_widget",
 *   label = @Translation("Paragraph Details"),
 *   description = @Translation("Extends Paragraphs Legacy (classic) widget by wrapping it in a details / accordion element."),
 *   field_types = {
 *     "entity_reference_revisions"
 *   }
 * )
 */
class ParagraphGroupDetailsWidget extends InlineParagraphsWidget implements ContainerFactoryPluginInterface {

  /**
   * The paragraph group helper service.
   *
   * @var \Drupal\paragraph_group\Paragroup\ParagroupHelperService
   */
  protected $helperService;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $entityTypeBundleInfo;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {

    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->helperService = $container->get('paragraph_group.helper');
    $instance->entityTypeBundleInfo = $container->get('entity_type.bundle.info');

    return $instance;

  }

  /**
   * Gets the non machine-name label / name for a Paragraph.
   *
   * @param string $machine_name
   *   The paragraph bundle machine name.
   *
   * @return string|false
   *   The paragraph type label, or FALSE if not found.
   */
  private function getParagraphLabel(string $machine_name): string|false {

    $paragraph_types =
      $this->entityTypeBundleInfo
        ->getBundleInfo('paragraph');

    /** @var array<string, array{label?: string}> $paragraph_types */
    if (isset($paragraph_types[$machine_name]['label'])) {
      return $paragraph_types[$machine_name]['label'];
    }

    return FALSE;

  }

  /**
   * Retrieves the typed widget state for a field.
   *
   * @param array<string, mixed> $element
   *   The form element array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return array{
   *   paragraphs?: array<int, array{
   *     entity?: \Drupal\paragraphs\Entity\Paragraph
   *   }>
   *   }
   *   The typed widget state array.
   */
  private function getTypedWidgetState(
    array $element,
    FormStateInterface $form_state,
  ): array {

    $parents = $element['#field_parents'];
    $field_name = $this->fieldDefinition->getName();

    if (!is_array($parents)) {
      $parents = [];
    }

    $widget_state = static::getWidgetState($parents, $field_name, $form_state);

    /**
     * @var array{
     *   paragraphs?: array<int, array{
     *     entity?: \Drupal\paragraphs\Entity\Paragraph
     *   }>
     * } $widget_state
     */

    return $widget_state;

  }

  /**
   * Gets the Summary text for a Details element.
   *
   * Ensures that the summary is fewer than 256 characters in length.
   *
   * @param array<string, mixed> $element
   *   The form element array.
   * @param int $delta
   *   The paragraph delta.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return string|null
   *   The summary text, or NULL if none found.
   */
  private function getSummaryText(
    array $element,
    int $delta,
    FormStateInterface $form_state,
  ): ?string {

    $widget_state = $this->getTypedWidgetState($element, $form_state);

    if (isset($widget_state['paragraphs'][$delta]['entity'])) {

      $paragraphs_entity = $widget_state['paragraphs'][$delta]['entity'];
      $summary_items = $paragraphs_entity->getSummaryItems();

      /** @var array{content?: array<int, string>} $summary_items */
      if (isset($summary_items['content'])) {

        foreach ($summary_items['content'] as $item) {
          if (mb_strlen($item) < 256) {
            return $item;
          }
        }

      }

    }

    return NULL;

  }

  /**
   * Returns the prefixed Summary text.
   *
   * @param array<string, mixed> $element
   *   The form element array.
   * @param int $delta
   *   The paragraph delta.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The formatted summary text or 'Paragraph' if no summary.
   */
  private function getDetailSummary(
    array $element,
    int $delta,
    FormStateInterface $form_state,
  ): TranslatableMarkup {

    $summary = $this->getSummaryText($element, $delta, $form_state);

    if ($summary) {
      return $this->t('Summary: @summary', ['@summary' => $summary]);
    }

    return $this->t('Paragraph');

  }

  /**
   * Overrides the Element data used by the formElement function.
   *
   * @param array<string, mixed> $element
   *   The form element array.
   * @param int $delta
   *   The paragraph delta.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   *
   * @return array<string, mixed>
   *   The modified element array with details formatting.
   */
  private function getElementData(
    array $element,
    int $delta,
    FormStateInterface $form_state,
  ): array {

    $paragraph_type = $element['#paragraph_type'];

    if (!is_string($paragraph_type)) {
      $paragraph_type = '';
    }

    $paragraph_label = $this->getParagraphLabel($paragraph_type);
    $type_text = "Paragraph Type: " . $paragraph_label;

    $detail_summary =
      $this->getDetailSummary($element, $delta, $form_state);

    $element['#type'] = 'details';
    $element['#title'] = $detail_summary;

    if (is_string($element['#prefix'])) {
      $element['#prefix'] .= '<span>' . Html::escape($type_text) . '</span>';
    }
    else {
      $element['#prefix'] = '<span>' . Html::escape($type_text) . '</span>';
    }

    return $element;

  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {

    // Remove edit mode from summary.
    $old_summary = parent::settingsSummary();
    $new_summary = [];

    foreach ($old_summary as $key => $val) {

      if ($key != 2) {
        $new_summary[] = $val;
      }

    }

    return $new_summary;

  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(
    array $form,
    FormStateInterface $form_state,
  ): array {

    $elements = parent::settingsForm($form, $form_state);

    // Remove edit mode from settings form.
    if (isset($elements['edit_mode']) && is_array($elements['edit_mode'])) {
      $elements['edit_mode']['#type'] = 'hidden';
    }

    return $elements;

  }

  /**
   * {@inheritdoc}
   */
  public function formElement(
    FieldItemListInterface $items,
    $delta,
    array $element,
    array &$form,
    FormStateInterface $form_state,
  ): array {

    $element =
      parent::formElement($items, $delta, $element, $form, $form_state);

    // Setup form elements as details html elements:
    // Cast to proper type for getElementData.
    /** @var array<string, mixed> $typed_element */
    $typed_element = [];

    foreach ($element as $key => $value) {
      $typed_element[(string) $key] = $value;
    }

    $element = $this->getElementData($typed_element, $delta, $form_state);

    // Remove paragraph type info from within paragraph form element,
    // as just above we've added it to the prefix outside the details widget.
    /** @var array{top?: array{paragraph_type_title?: array{info?: mixed}}} $element */
    if (isset($element['top']['paragraph_type_title']['info'])) {
      unset($element['top']['paragraph_type_title']['info']);
    }

    return $element;

  }

  /**
   * {@inheritdoc}
   */
  public function formMultipleElements(
    FieldItemListInterface $items,
    array &$form,
    FormStateInterface $form_state,
  ): array {

    // The Paragraph Details widget only supports the default 'open' edit mode,
    // and improves it so that the other edit modes shouldn't be necessary.
    // So setting all form elements edit_mode to 'open'.
    $this->setSetting('edit_mode', 'open');

    $widget = parent::formMultipleElements($items, $form, $form_state);

    /** @var array<string, mixed> $widget */
    if (isset($widget['#attached'])) {

      $attached = &$widget['#attached'];

      if (is_array($attached)) {

        /** @var array<string, mixed> $attached */
        $this->helperService->attachDetailsWidget($attached);

      }

    }

    return $widget;

  }

}
