<?php

declare(strict_types=1);

namespace Drupal\paragraph_lineage\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Paragraph Lineage settings for this site.
 */
final class ParagraphLineageSettingsForm extends ConfigFormBase {

  /**
   * The paragraph lineage settings service.
   *
   * @var \Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface
   */
  private ParagraphLineageSettingsServiceInterface $settingsService;

  /**
   * Constructs a ParagraphLineageSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed configuration manager.
   * @param \Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface $settingsService
   *   The paragraph lineage settings service.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    ParagraphLineageSettingsServiceInterface $settingsService,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
    $this->settingsService = $settingsService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): ParagraphLineageSettingsForm {
    /** @var \Drupal\Core\Config\ConfigFactoryInterface $configFactory */
    $configFactory = $container->get('config.factory');

    /** @var \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager */
    $typedConfigManager = $container->get('config.typed');

    /** @var \Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface $settingsService */
    $settingsService = $container->get('paragraph_lineage.settings');

    return new ParagraphLineageSettingsForm(
      $configFactory,
      $typedConfigManager,
      $settingsService,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'paragraph_lineage_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['paragraph_lineage.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state):array {

    $form['preferred_theme'] = [
      '#type' => 'radios',
      '#title' => $this->t('Preferred Theme'),
      '#description' => $this->t('Select the preferred theme for displaying paragraph lineage.'),
      '#options' => [
        'default' => $this->t('Default Theme'),
        'admin' => $this->t('Admin Theme'),
      ],
      '#required' => TRUE,
      '#default_value' => $this->settingsService->getPreferredThemeId(TRUE),
    ];
    if ($this->settingsService->isPreferredThemeOverridden()) {
      $this->messenger()->addWarning($this->t("The setting for the preferred theme for paragraph lineage pages is currently set in settings.php to '@theme'.", [
        '@theme' => $this->settingsService->getPreferredThemeId(),
      ]));
      $form['preferred_theme']['#description'] = $this->t('This setting is overridden as <code><strong>@setting</strong></code> in settings.php and cannot be changed here.', [
        '@setting' => $this->settingsService->getPreferredThemeId(),
      ]);
      $form['preferred_theme']['#disabled'] = TRUE;
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->settingsService->setPreferredThemeId($form_state->getValue('preferred_theme'));
    parent::submitForm($form, $form_state);
  }

}
