<?php

namespace Drupal\paragraph_lineage\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'paragraph_lineage_paragraph_id_link' formatter.
 *
 * @FieldFormatter(
 *   id = "paragraph_lineage_paragraph_id_link",
 *   label = @Translation("Link to Paragraph"),
 *   field_types = {
 *     "integer",
 *     "number"
 *   }
 * )
 */
class ParagraphIdLinkFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation */
    $stringTranslation = $container->get('string_translation');
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $stringTranslation,
    );
  }

  /**
   * Constructs a ParagraphIdLinkFormatter object.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Third party settings.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    TranslationInterface $string_translation,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->setStringTranslation($string_translation);
  }

  /**
   * Builds a renderable array of links for each field item.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   The field values to be displayed.
   * @param string $langcode
   *   The language code for rendering.
   *
   * @return array
   *   A renderable array of link elements.
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $elements = [];

    foreach ($items as $delta => $item) {
      $entity = $item->getEntity();
      $label = $this->t('View @name', ['@name' => $entity->label()]);
      if ($entity->getEntityTypeId() === 'paragraph') {
        $label = $this->t('View lineage for @name', ['@name' => $entity->label()]);
      }
      if ($item->value) {
        // Create a link to the custom route using the paragraph ID.
        try {
          $url = $entity->toUrl();
        }
        catch (\Exception $e) {
          continue;
        }

        // Render the link.
        $elements[$delta] = [
          '#type' => 'link',
          '#title' => $label,
          '#url' => $url,
        ];
      }
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    return [];
  }

}
