<?php

namespace Drupal\Tests\paragraph_lineage\Functional;

use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\Tests\BrowserTestBase;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Core\Url;

/**
 * Tests access control for paragraph lineage routes.
 *
 * @group paragraph_lineage
 */
class ParagraphLineageAccessTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'user',
    'paragraphs',
    'paragraph_lineage',
    'field',
    'text',
  ];

  /**
   * A test paragraph.
   *
   * @var \Drupal\paragraphs\Entity\Paragraph
   */
  protected Paragraph $paragraph;

  /**
   * A test node.
   *
   * @var \Drupal\node\Entity\Node
   */
  protected Node $node;

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a paragraph type.
    $paragraph_type = ParagraphsType::create([
      'id' => 'test_paragraph',
      'label' => 'Test Paragraph',
    ]);
    $paragraph_type->save();

    // Create a content type.
    $node_type = NodeType::create([
      'type' => 'test_content',
      'name' => 'Test Content',
    ]);
    $node_type->save();

    // Create a paragraph field on the content type.
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_paragraphs',
      'entity_type' => 'node',
      'type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'paragraph',
      ],
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'test_content',
      'settings' => [
        'handler' => 'default:paragraph',
        'handler_settings' => [
          'target_bundles' => ['test_paragraph'],
        ],
      ],
    ]);
    $field->save();

    // Create a test paragraph.
    $this->paragraph = Paragraph::create([
      'type' => 'test_paragraph',
    ]);
    $this->paragraph->save();

    // Create a test node with the paragraph.
    $this->node = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node',
      'field_paragraphs' => [
        [
          'target_id' => $this->paragraph->id(),
          'target_revision_id' => $this->paragraph->getRevisionId(),
        ],
      ],
    ]);
    $this->node->save();
  }

  /**
   * Tests that anonymous users cannot access paragraph lineage routes.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testAnonymousUserAccessDenied(): void {
    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $this->paragraph->id()]);
    $this->drupalGet($url);
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that authenticated users without permission cannot access routes.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testAuthenticatedUserWithoutPermissionAccessDenied(): void {
    $user = $this->drupalCreateUser([]);
    $this->drupalLogin($user);

    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $this->paragraph->id()]);
    $this->drupalGet($url);
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that users with 'access administration pages' can access routes.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testUserWithAdminPermissionCanAccess(): void {
    $user = $this->drupalCreateUser(['access administration pages']);
    $this->drupalLogin($user);

    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $this->paragraph->id()]);
    $this->drupalGet($url);
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Test Node');
  }

  /**
   * Tests access with admin user role.
   */
  public function testAdminUserCanAccess(): void {
    $admin_user = $this->drupalCreateUser([
      'access administration pages',
      'administer content types',
      'administer nodes',
    ]);
    $this->drupalLogin($admin_user);

    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $this->paragraph->id()]);
    $this->drupalGet($url);
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests 404 response for non-existent paragraph.
   */
  public function testNonExistentParagraphReturns404(): void {
    $user = $this->drupalCreateUser(['access administration pages']);
    $this->drupalLogin($user);

    $non_existent_id = 99999;
    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $non_existent_id]);
    $this->drupalGet($url);
    $this->assertSession()->statusCodeEquals(404);
  }

  /**
   * Tests access with invalid paragraph ID format.
   */
  public function testInvalidParagraphIdFormat(): void {
    $user = $this->drupalCreateUser(['access administration pages']);
    $this->drupalLogin($user);

    $url = "/admin/content/paragraph-lineage/paragraph/invalid-id";
    $this->drupalGet($url);
    $this->assertSession()->statusCodeEquals(404);
  }

  /**
   * Tests that the route is properly marked as an admin route.
   */
  public function testRouteIsAdminRoute(): void {
    $user = $this->drupalCreateUser(['access administration pages']);
    $this->drupalLogin($user);

    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $this->paragraph->id()]);
    $this->drupalGet($url);

    // Admin routes typically have admin themes and specific CSS classes.
    $this->assertSession()->statusCodeEquals(200);
    // Verify we're in an admin context by checking for admin-specific elements.
    $this->assertSession()->elementExists('css', 'body');
  }

  /**
   * Tests access to orphaned paragraph (no parent entity).
   */
  public function testOrphanedParagraphAccess(): void {
    // Create a user with permission to access administration pages.
    $user = $this->drupalCreateUser(['access administration pages']);
    $this->drupalLogin($user);

    // Create an orphaned paragraph of type 'test_paragraph'.
    // An orphaned paragraph is one that is not attached to any parent entity.
    $orphaned_paragraph = Paragraph::create([
      'type' => 'test_paragraph',
    ]);
    $orphaned_paragraph->save();

    // Build the URL for the paragraph lineage route using the orphaned
    // paragraph ID.
    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $this->paragraph->id()]);
    $this->drupalGet($url);

    // Assert that the page loads successfully (HTTP 200).
    // Even though the paragraph has no parent entity, the route should still
    // render.
    $this->assertSession()->statusCodeEquals(200);

  }

}
