<?php

declare(strict_types=1);

namespace Drupal\Tests\paragraph_lineage\Functional;

use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\Tests\BrowserTestBase;
use Drupal\Core\Url;

/**
 * Tests the paragraph lineage theme settings functionality.
 *
 * @group paragraph_lineage
 */
class ParagraphLineageThemeSettingsTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'olivero';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'paragraph_lineage',
    'paragraphs',
    'user',
    'system',
  ];

  /**
   * A user with administrative permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create an admin user.
    $this->adminUser = $this->drupalCreateUser([
      'access administration pages',
    ]);

    // Create a paragraph type for testing.
    $paragraph_type = ParagraphsType::create([
      'id' => 'test_paragraph',
      'label' => 'Test Paragraph',
    ]);
    $paragraph_type->save();
  }

  /**
   * Tests the paragraph lineage settings form.
   */
  public function testParagraphLineageSettingsForm(): void {
    $this->drupalLogin($this->adminUser);

    // Visit the settings form.
    $this->drupalGet(Url::fromRoute('paragraph_lineage.settings'));
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Paragraph lineage settings');
    $this->assertSession()->pageTextContains('Paragraph lineage settings');

    // Check that the form elements are present.
    $this->assertSession()->elementExists('css', 'input[name="preferred_theme"]');
    $this->assertSession()->pageTextContains('Preferred Theme');
    $this->assertSession()->pageTextContains('Select the preferred theme for displaying paragraph lineage.');

    // Check that the radio options are available.
    $this->assertSession()->elementExists('css', 'input[name="preferred_theme"][value="default"]');
    $this->assertSession()->elementExists('css', 'input[name="preferred_theme"][value="admin"]');

    // Test form submission with 'admin' theme.
    $this->submitForm([
      'preferred_theme' => 'admin',
    ], 'Save configuration');

    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify the setting was saved.
    $config = $this->config('paragraph_lineage.settings');
    $this->assertEquals('admin', $config->get('preferred_theme'));

    // Test form submission with 'default' theme.
    $this->submitForm([
      'preferred_theme' => 'default',
    ], 'Save configuration');

    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify the setting was updated.
    $config = $this->config('paragraph_lineage.settings');
    $this->assertEquals('default', $config->get('preferred_theme'));
  }

  /**
   * Tests the settings form menu link.
   */
  public function testSettingsFormMenuLink(): void {
    $this->drupalLogin($this->adminUser);

    // Visit the content configuration page.
    $this->drupalGet('/admin/config/content');
    $this->assertSession()->statusCodeEquals(200);

    // Check that the menu link is present.
    $this->assertSession()->linkExists('Paragraph lineage settings');
    $this->assertSession()->pageTextContains('Change settings for paragraph lineage');

    // Click the link and verify it goes to the correct page.
    $this->clickLink('Paragraph lineage settings');
    // Get the expected url of the settings page.
    $expected_url = Url::fromRoute('paragraph_lineage.settings')->toString();
    // Remove everything before /admin in the URL.
    $expected_url = str_replace('/web', '', $expected_url);
    $this->assertSession()->addressEquals($expected_url);
    $this->assertSession()->pageTextContains('Paragraph lineage settings');
  }

  /**
   * Tests theme negotiation for paragraph canonical route.
   */
  public function testThemeNegotiationForParagraphRoute(): void {
    $this->drupalLogin($this->adminUser);

    // Create a test paragraph.
    $paragraph = Paragraph::create([
      'type' => 'test_paragraph',
    ]);
    $paragraph->save();

    // Set preferred theme to admin.
    $this->config('paragraph_lineage.settings')
      ->set('preferred_theme', 'admin')
      ->save();

    // Visit the paragraph canonical route.
    $this->drupalGet("/admin/content/paragraph-lineage/paragraph/{$paragraph->id()}");

    // The page should load successfully.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('View Paragraph');

    // Set preferred theme to default.
    $this->config('paragraph_lineage.settings')
      ->set('preferred_theme', 'default')
      ->save();

    // Visit the paragraph canonical route again.
    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $paragraph->id()]);
    $this->drupalGet($url);

    // The page should still load successfully.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('View Paragraph');
  }

  /**
   * Tests access control for the settings form.
   */
  public function testSettingsFormAccessControl(): void {
    // Create a user without admin permissions.
    $regular_user = $this->drupalCreateUser(['access content']);
    $this->drupalLogin($regular_user);
    $settings_url = Url::fromRoute('paragraph_lineage.settings');

    // Try to access the settings form.
    $this->drupalGet($settings_url);
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests the service integration.
   */
  public function testServiceIntegration(): void {
    /** @var \Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface $service */
    $service = \Drupal::service('paragraph_lineage.settings');

    // Test default theme retrieval.
    $default_theme = $service->getPreferredThemeId();
    $this->assertIsString($default_theme);

    // Test setting a custom theme.
    $service->setPreferredThemeId('custom_theme');
    $this->assertEquals('custom_theme', $service->getPreferredThemeId());

    // Test setting back to admin theme.
    $service->setPreferredThemeId('admin');
    $this->assertEquals('admin', $service->getPreferredThemeId());
  }

}
