<?php

declare(strict_types=1);

namespace Drupal\Tests\paragraph_lineage\Kernel;

use Drupal\Core\Routing\RouteMatch;
use Drupal\KernelTests\KernelTestBase;
use Symfony\Component\Routing\Route;

/**
 * Tests the paragraph lineage theme negotiator integration.
 *
 * @group paragraph_lineage
 */
class ParagraphLineageThemeNegotiatorIntegrationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'paragraph_lineage',
    'system',
    'user',
    'views',
  ];

  /**
   * The theme negotiator service.
   *
   * @var \Drupal\paragraph_lineage\Theme\ParagraphLineageThemeNegotiator
   */
  protected $themeNegotiator;

  /**
   * The settings service.
   *
   * @var \Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface
   */
  protected $settingsService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installConfig(['system', 'paragraph_lineage']);

    $this->themeNegotiator = $this->container->get('paragraph_lineage.theme.negotiator');
    $this->settingsService = $this->container->get('paragraph_lineage.settings');

  }

  /**
   * Tests theme negotiator applies to correct routes.
   */
  public function testThemeNegotiatorApplies(): void {
    $route = new Route('/test');
    $parameters = [];

    // Test with paragraph canonical route.
    $route_match = new RouteMatch('entity.paragraph.canonical', $route, $parameters);
    $this->assertTrue($this->themeNegotiator->applies($route_match));

    // Test with other routes.
    $route_match = new RouteMatch('some.other.route', $route, $parameters);
    $this->assertFalse($this->themeNegotiator->applies($route_match));
  }

  /**
   * Tests theme negotiator determines correct theme.
   */
  public function testThemeNegotiatorDeterminesTheme(): void {
    $route = new Route('/test');
    $parameters = [];
    $route_match = new RouteMatch('entity.paragraph.canonical', $route, $parameters);

    // Test with admin theme preference.
    $this->settingsService->setPreferredThemeId('admin');
    $theme = $this->themeNegotiator->determineActiveTheme($route_match);
    // Should return the admin theme from system configuration.
    $expected_admin_theme = $this->config('system.theme')->get('admin');
    $this->assertEquals($expected_admin_theme, $theme);

    // Test with default theme preference.
    $this->settingsService->setPreferredThemeId('default');
    $theme = $this->themeNegotiator->determineActiveTheme($route_match);
    // Should return the default theme from system configuration.
    $expected_default_theme = $this->config('system.theme')->get('default');
    $this->assertEquals($expected_default_theme, $theme);

    // Test with custom theme preference.
    $this->settingsService->setPreferredThemeId('custom_theme');
    $theme = $this->themeNegotiator->determineActiveTheme($route_match);
    $this->assertEquals('custom_theme', $theme);
  }

  /**
   * Tests settings service configuration persistence.
   */
  public function testSettingsServiceConfigurationPersistence(): void {
    // Test initial state.
    $initial_theme = $this->settingsService->getPreferredThemeId();
    $this->assertIsString($initial_theme);

    // Test setting and getting custom theme.
    $this->settingsService->setPreferredThemeId('test_theme');
    $this->assertEquals('test_theme', $this->settingsService->getPreferredThemeId());

    // Test configuration persistence by recreating the service.
    $new_settings_service = $this->container->get('paragraph_lineage.settings');
    $this->assertEquals('test_theme', $new_settings_service->getPreferredThemeId());

    // Test setting different theme types.
    $this->settingsService->setPreferredThemeId('admin');
    $this->assertEquals('admin', $this->settingsService->getPreferredThemeId());

    $this->settingsService->setPreferredThemeId('default');
    $this->assertEquals('default', $this->settingsService->getPreferredThemeId());
  }

  /**
   * Tests fallback behavior when no theme is configured.
   */
  public function testFallbackBehavior(): void {
    // Clear the configuration to test fallback.
    $this->config('paragraph_lineage.settings')->delete();
    $this->container->get('config.factory')->clearStaticCache();

    // The service should fall back to literal 'admin' value.
    $theme = $this->settingsService->getPreferredThemeId();
    $this->assertEquals('admin', $theme);
  }

  /**
   * Tests theme negotiator service registration.
   */
  public function testThemeNegotiatorServiceRegistration(): void {
    // Verify the service is properly registered.
    $this->assertTrue($this->container->has('paragraph_lineage.theme.negotiator'));
    $service = $this->container->get('paragraph_lineage.theme.negotiator');
    $this->assertInstanceOf('Drupal\paragraph_lineage\Theme\ParagraphLineageThemeNegotiator', $service);

    // Verify the settings service is properly registered.
    $this->assertTrue($this->container->has('paragraph_lineage.settings'));
    $settings_service = $this->container->get('paragraph_lineage.settings');
    $this->assertInstanceOf('Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface', $settings_service);
  }

}
