<?php

namespace Drupal\Tests\paragraph_lineage\Unit\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Url;
use Drupal\paragraph_lineage\Plugin\Field\FieldFormatter\ParagraphIdLinkFormatter;
use Drupal\Tests\UnitTestCase;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * Tests the ParagraphIdLinkFormatter.
 *
 * @group paragraph_lineage
 * @coversDefaultClass \Drupal\paragraph_lineage\Plugin\Field\FieldFormatter\ParagraphIdLinkFormatter
 */
class ParagraphIdLinkFormatterTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The formatter being tested.
   *
   * @var \Drupal\paragraph_lineage\Plugin\Field\FieldFormatter\ParagraphIdLinkFormatter
   */
  protected $formatter;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->stringTranslation = $this->getStringTranslationStub();

    // Create a mock field definition.
    $field_definition = $this->prophesize(FieldDefinitionInterface::class);
    $field_definition->getSettings()->willReturn([]);
    $field_definition->getType()->willReturn('integer');

    $this->formatter = new ParagraphIdLinkFormatter(
      'paragraph_id_link',
      [],
      $field_definition->reveal(),
      [],
      'default',
      'default',
      [],
      $this->stringTranslation
    );
  }

  /**
   * @covers ::viewElements
   */
  public function testViewElementsWithParagraphEntity() {
    // Create a mock field item list.
    $field_item_list = $this->prophesize(FieldItemListInterface::class);

    // Create a mock paragraph entity.
    $entity = $this->prophesize(EntityInterface::class);
    $entity->getEntityTypeId()->willReturn('paragraph');
    $entity->label()->willReturn('Test Paragraph');

    // Mock the toUrl method to return a URL object.
    $url = $this->prophesize(Url::class);
    $entity->toUrl()->willReturn($url->reveal());

    // Create a mock field item with custom class for magic property access.
    $field_item = new MockFieldItem(123, $entity->reveal());

    // Set up the field item list to return our mock field item.
    $field_item_list->willImplement(\IteratorAggregate::class);
    $field_item_list->getIterator()
      ->willReturn(new \ArrayIterator([$field_item]));

    // Run the formatter.
    $elements = $this->formatter->viewElements($field_item_list->reveal(),
      'en');

    // Assert the link was created correctly.
    $this->assertEquals('link', $elements[0]['#type']);
    $this->assertStringContainsString('View lineage for Test Paragraph',
      (string) $elements[0]['#title']);
    $this->assertInstanceOf(Url::class, $elements[0]['#url']);
  }

  /**
   * @covers ::viewElements
   */
  public function testViewElementsWithNonParagraphEntity() {
    // Create a mock field item list.
    $field_item_list = $this->prophesize(FieldItemListInterface::class);

    // Create a mock non-paragraph entity.
    $entity = $this->prophesize(EntityInterface::class);
    $entity->getEntityTypeId()->willReturn('node');
    $entity->label()->willReturn('Test Node');

    // Mock the toUrl method to return a URL object.
    $url = $this->prophesize(Url::class);
    $entity->toUrl()->willReturn($url->reveal());

    // Create a mock field item. The custom test class allows access to
    // properties dynamically, leveraging PHP's magic methods like __get().
    $field_item = new MockFieldItem(456, $entity->reveal());

    // Set up the field item list to return our mock field item.
    $field_item_list->willImplement(\IteratorAggregate::class);
    $field_item_list->getIterator()
      ->willReturn(new \ArrayIterator([$field_item]));

    // Run the formatter.
    $elements = $this->formatter->viewElements($field_item_list->reveal(), 'en');

    // Assert the link was created correctly.
    $this->assertEquals('link', $elements[0]['#type']);
    $this->assertStringContainsString('View Test Node', (string) $elements[0]['#title']);
    $this->assertInstanceOf(Url::class, $elements[0]['#url']);
  }

  /**
   * @covers ::viewElements
   */
  public function testViewElementsWithEmptyValue() {
    // Create a mock field item list.
    $field_item_list = $this->prophesize(FieldItemListInterface::class);

    // Create a mock entity.
    $entity = $this->prophesize(EntityInterface::class);
    $entity->getEntityTypeId()->willReturn('paragraph');
    $entity->label()->willReturn('Test Paragraph');

    // Create a mock field item with empty value, using a custom test class to
    // handle the magic property access.
    $field_item = new MockFieldItem(0, $entity->reveal());

    // Set up the field item list to return our mock field item.
    $field_item_list->willImplement(\IteratorAggregate::class);
    $field_item_list->getIterator()
      ->willReturn(new \ArrayIterator([$field_item]));

    // Run the formatter.
    $elements = $this->formatter->viewElements($field_item_list->reveal(), 'en');

    // Assert no elements were created.
    $this->assertEmpty($elements);
  }

}
