<?php

declare(strict_types=1);

namespace Drupal\paragraph_lineage\Controller;

use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\paragraphs\ParagraphInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns responses for Paragraph Lineage routes.
 */
class ParagraphLineageViewController extends ControllerBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * The entity display repository service.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  private EntityDisplayRepositoryInterface $entityDisplayRepository;

  /**
   * Constructs a ParagraphViewController.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository service.
   */
  public function __construct(
    RendererInterface $renderer,
    RouteMatchInterface $route_match,
    EntityDisplayRepositoryInterface $entity_display_repository,
  ) {
    $this->renderer = $renderer;
    $this->routeMatch = $route_match;
    $this->entityDisplayRepository = $entity_display_repository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): ParagraphLineageViewController {
    /** @var \Drupal\Core\Render\RendererInterface $renderer */
    $renderer = $container->get('renderer');
    /** @var \Drupal\Core\Routing\RouteMatchInterface $route_match */
    $route_match = $container->get('current_route_match');
    /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository */
    $entity_display_repository = $container->get('entity_display.repository');
    return new ParagraphLineageViewController(
      $renderer,
      $route_match,
      $entity_display_repository
    );
  }

  /**
   * Builds the response.
   *
   * @throws \Exception
   */
  public function build(): array {
    // Get the paragraph from the route.
    /** @var \Drupal\paragraphs\ParagraphInterface $paragraph */
    $paragraph = $this->routeMatch->getParameter('paragraph');
    if (!$paragraph instanceof ParagraphInterface) {
      throw new \InvalidArgumentException('No paragraph found in the route.');
    }

    // Determine if a teaser or preview mode is identified for the
    // $paragraph->bundle().
    $rendered_paragraph = $this->renderEntityPreview($paragraph);

    $lineage = [];
    // Get the parent of the entity.
    $parent = $paragraph->getParentEntity();
    if ($parent instanceof EntityInterface) {
      $this->getLineage($lineage, $parent);
    }
    return [
      '#theme' => 'paragraph_lineage',
      '#paragraph' => [
        'type' => $paragraph->getEntityTypeId(),
        'bundle' => $paragraph->bundle(),
        'label' => $paragraph->label(),
        'content' => $rendered_paragraph,
      ],
      '#lineage' => $lineage,
    ];
  }

  /**
   * Recursively retrieves the lineage of an entity.
   *
   * This function generates a renderable array of links representing the
   * hierarchy of parent entities for a given entity. If the entity is a
   * paragraph, it traverses up the hierarchy to include all parent entities.
   * For non-paragraph entities, it simply retrieves the canonical route link.
   *
   * @param array &$lineage
   *   A renderable array of the lineage of the entity.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity for which to retrieve the lineage.
   *
   * @throws \Exception
   */
  protected function getLineage(array &$lineage, EntityInterface $entity): void {
    $route = 'entity.' . $entity->getEntityTypeId() . '.canonical';
    $parent = NULL;
    if ($entity instanceof ParagraphInterface) {
      $route = 'entity.paragraph.canonical';
      $parent = $entity->getParentEntity();
    }

    $lineage[] = [
      'type' => $entity->getEntityTypeId(),
      'bundle' => $entity->bundle(),
      'content' => $this->renderEntityPreview($entity),
      'link' => Link::createFromRoute($entity->label(), $route, [$entity->getEntityTypeId() => $entity->id()])
        ->toRenderable(),
    ];
    if ($parent instanceof EntityInterface) {
      $this->getLineage($lineage, $parent);
    }

  }

  /**
   * Renders an entity in a preview mode if available.
   *
   * @param \Drupal\paragraphs\ParagraphInterface $entity
   *   The paragraph entity to render.
   *
   * @return \Drupal\Component\Render\MarkupInterface
   *
   *   This function renders an entity in a preview mode if available. If a
   *   preview mode is not available, it falls back to the default view mode.
   *
   * @throws \Exception
   */
  protected function renderEntityPreview(EntityInterface $entity): MarkupInterface {
    $view_mode = 'default';
    $view_modes = $this->entityDisplayRepository->getViewModes($entity->bundle());
    // Check if preview or teaser are available.
    // If so, set that as the view mode.
    if (isset($view_modes['preview'])) {
      $view_mode = 'preview';
    }
    else {
      if (isset($view_modes['teaser'])) {
        $view_mode = 'teaser';
      }
    }

    // Render the entity using the defined view mode.
    $view = $this->entityTypeManager()
      ->getViewBuilder($entity->getEntityTypeId())
      ->view($entity, $view_mode);
    return $this->renderer->render($view);
  }

}
