<?php

namespace Drupal\paragraph_lineage\Plugin\views\field;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Url;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A custom views field that provides a link to paragraph lineage.
 *
 * @ViewsField("paragraph_lineage_file_usage_paragraph_link")
 */
class FileUsageParagraphLink extends FieldPluginBase {

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $loggerChannel;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, LoggerChannelFactoryInterface $loggerFactory) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->loggerChannel = $loggerFactory->get('paragraph_lineage');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): FileUsageParagraphLink {
    // Ensure the logger channel is set up.
    /** @var \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory */
    $loggerFactory = $container->get('logger.factory');
    return new FileUsageParagraphLink(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $loggerFactory,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values): array {
    $entity_id = $this->getValue($values);

    if (empty($entity_id)) {
      return [];
    }

    try {
      $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $entity_id]);
      return [
        '#type' => 'link',
        '#title' => $this->t('View lineage'),
        '#url' => $url,
        '#attributes' => [
          'class' => ['paragraph-lineage-link'],
        ],
      ];
    }
    catch (\Exception $e) {
      // Log the error if the URL cannot be created.
      $this->loggerChannel->error('Error creating URL for paragraph with ID @id: @message', [
        '@id' => $entity_id,
        '@message' => $e->getMessage(),
      ]);
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    // This field doesn't need to add anything to the query.
  }

}
