<?php

namespace Drupal\paragraph_lineage\Theme;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Theme\ThemeNegotiatorInterface;
use Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface;

/**
 * Negotiates theme for the Paragraph Lineage route.
 *
 * This class determines which theme should be used when displaying
 * paragraph lineage pages.
 */
class ParagraphLineageThemeNegotiator implements ThemeNegotiatorInterface {

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * The paragraph lineage settings service.
   *
   * @var \Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface
   */
  protected ParagraphLineageSettingsServiceInterface $settingsService;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a ParagraphLineageThemeNegotiator.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The route match service.
   * @param \Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface $settingsService
   *   The settings service for paragraph lineage.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   */
  public function __construct(RouteMatchInterface $routeMatch, ParagraphLineageSettingsServiceInterface $settingsService, ConfigFactoryInterface $configFactory) {
    $this->routeMatch = $routeMatch;
    $this->settingsService = $settingsService;
    $this->configFactory = $configFactory;
  }

  /**
   * {@inheritdoc}
   */
  public function applies(RouteMatchInterface $route_match): bool {
    return $route_match->getRouteName() === 'entity.paragraph.canonical';
  }

  /**
   * {@inheritdoc}
   */
  public function determineActiveTheme(RouteMatchInterface $route_match): ?string {
    $theme = $this->settingsService->getPreferredThemeId();
    if (in_array($theme, ['default', 'admin'], TRUE)) {
      return $this->configFactory->get('system.theme')->get($theme);
    }
    return $theme;
  }

}
