<?php

namespace Drupal\Tests\paragraph_lineage\Functional;

use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\Tests\BrowserTestBase;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\user\UserInterface;
use Drupal\Core\Url;

/**
 * Tests the paragraph lineage route functionality.
 *
 * @group paragraph_lineage
 */
class ParagraphLineageRouteTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'user',
    'paragraphs',
    'paragraph_lineage',
    'field',
    'text',
  ];

  /**
   * A test user with admin permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $adminUser;

  /**
   * A test paragraph.
   *
   * @var \Drupal\paragraphs\Entity\Paragraph
   */
  protected Paragraph $paragraph;

  /**
   * A test node.
   *
   * @var \Drupal\node\Entity\Node
   */
  protected Node $node;

  /**
   * A nested paragraph for testing hierarchy.
   *
   * @var \Drupal\paragraphs\Entity\Paragraph
   */
  protected Paragraph $nestedParagraph;

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function setUp(): void {
    parent::setUp();

    // Create admin user.
    $this->adminUser = $this->drupalCreateUser([
      'access administration pages',
      'administer content types',
      'administer nodes',
    ]);

    // Create paragraph types.
    $paragraph_type = ParagraphsType::create([
      'id' => 'text_paragraph',
      'label' => 'Text Paragraph',
    ]);
    $paragraph_type->save();

    $container_type = ParagraphsType::create([
      'id' => 'container_paragraph',
      'label' => 'Container Paragraph',
    ]);
    $container_type->save();

    // Create content type.
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    // Create paragraph field on node.
    $this->createParagraphField('node', 'article', 'field_paragraphs');

    // Create nested paragraph field on container paragraph.
    $this->createParagraphField('paragraph', 'container_paragraph', 'field_nested_paragraphs');

    // Create nested paragraph structure:
    // Node -> Container Paragraph -> Text Paragraph.
    $this->nestedParagraph = Paragraph::create([
      'type' => 'text_paragraph',
    ]);
    $this->nestedParagraph->save();

    $this->paragraph = Paragraph::create([
      'type' => 'container_paragraph',
      'field_nested_paragraphs' => [
        [
          'target_id' => $this->nestedParagraph->id(),
          'target_revision_id' => $this->nestedParagraph->getRevisionId(),
        ],
      ],
    ]);
    $this->paragraph->save();

    $this->node = Node::create([
      'type' => 'article',
      'title' => 'Test Article Node',
      'field_paragraphs' => [
        [
          'target_id' => $this->paragraph->id(),
          'target_revision_id' => $this->paragraph->getRevisionId(),
        ],
      ],
    ]);
    $this->node->save();

    $this->drupalLogin($this->adminUser);
  }

  /**
   * Helper to create a paragraph field.
   */
  protected function createParagraphField(string $entity_type, string $bundle, string $field_name): void {
    $field_storage = FieldStorageConfig::create([
      'field_name' => $field_name,
      'entity_type' => $entity_type,
      'type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'paragraph',
      ],
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => $bundle,
      'settings' => [
        'handler' => 'default:paragraph',
        'handler_settings' => [
          'target_bundles' => ['text_paragraph', 'container_paragraph'],
        ],
      ],
    ]);
    $field->save();
  }

  /**
   * Tests that the route renders the paragraph lineage theme.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testRouteRendersTheme(): void {
    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $this->paragraph->id()]);
    $this->drupalGet($url);

    $this->assertSession()->statusCodeEquals(200);

    // Check that our custom theme template is being used.
    $this->assertSession()->responseContains('paragraph-lineage');

    // Verify paragraph information is displayed.
    $this->assertSession()->pageTextContains('container_paragraph');
  }

  /**
   * Tests that lineage hierarchy is correctly displayed.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testLineageHierarchyDisplay(): void {
    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $this->nestedParagraph->id()]);
    $this->drupalGet($url);

    $this->assertSession()->statusCodeEquals(200);

    // Should show the nested paragraph's lineage back to the node.
    $this->assertSession()->pageTextContains('Test Article Node');
    $this->assertSession()->pageTextContains('container_paragraph');
    $this->assertSession()->pageTextContains('text_paragraph');
  }

  /**
   * Tests route with paragraph that has no parent.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testOrphanedParagraphLineage(): void {
    $orphaned_paragraph = Paragraph::create([
      'type' => 'text_paragraph',
    ]);
    $orphaned_paragraph->save();

    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $orphaned_paragraph->id()]);
    $this->drupalGet($url);

    $this->assertSession()->statusCodeEquals(200);

    // Should render the paragraph but with no lineage.
    $this->assertSession()->pageTextContains('text_paragraph');

    // Verify the template is still used even without lineage.
    $this->assertSession()->responseContains('paragraph-lineage');
  }

  /**
   * Tests that links in lineage are properly generated.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testLineageLinksGenerated(): void {
    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $this->nestedParagraph->id()]);

    $this->drupalGet($url);

    $this->assertSession()->statusCodeEquals(200);

    // Check that links to parent entities are created.
    // The node should have a link to its canonical route.
    $node_link = Url::fromRoute('entity.node.canonical', ['node' => $this->node->id()])
      ->toString();
    $this->assertSession()->linkByHrefExists($node_link);

    // The container paragraph should have a link to its lineage route.
    $paragraph_link = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $this->paragraph->id()])
      ->toString();
    $this->assertSession()->linkByHrefExists($paragraph_link);
  }

  /**
   * Tests that entity view modes are respected.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testEntityViewModeHandling(): void {
    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $this->nestedParagraph->id()])->toString();
    $this->drupalGet($url);

    $this->assertSession()->statusCodeEquals(200);
    // The controller should render the paragraph with appropriate view mode.
    $this->assertSession()->pageTextContains('text_paragraph');
  }

  /**
   * Tests route parameter validation.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testRouteParameterValidation(): void {
    // Test with non-existent paragraph ID.
    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => 99999])->toString();

    $this->drupalGet($url);
    $this->assertSession()->statusCodeEquals(404);

    // Test with invalid paragraph ID format.
    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => 'invalid'])->toString();
    $this->drupalGet($url);
    $this->assertSession()->statusCodeEquals(404);
  }

  /**
   * Tests that the route title is correctly set.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   */
  public function testRouteTitle(): void {
    $url = "/admin/content/paragraph-lineage/paragraph/" . $this->paragraph->id();
    $this->drupalGet($url);

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->titleEquals('View Paragraph | Drupal');
  }

  /**
   * Tests complex nested paragraph hierarchy.
   *
   * @throws \Behat\Mink\Exception\ExpectationException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testDeepNestedHierarchy(): void {
    // Create a deeper nesting: Node -> Container -> Container -> Text.
    $deep_text = Paragraph::create(['type' => 'text_paragraph']);
    $deep_text->save();

    $mid_container = Paragraph::create([
      'type' => 'container_paragraph',
      'field_nested_paragraphs' => [
        [
          'target_id' => $deep_text->id(),
          'target_revision_id' => $deep_text->getRevisionId(),
        ],
      ],
    ]);
    $mid_container->save();

    $top_container = Paragraph::create([
      'type' => 'container_paragraph',
      'field_nested_paragraphs' => [
        [
          'target_id' => $mid_container->id(),
          'target_revision_id' => $mid_container->getRevisionId(),
        ],
      ],
    ]);
    $top_container->save();

    $deep_node = Node::create([
      'type' => 'article',
      'title' => 'Deep Nested Node',
      'field_paragraphs' => [
        [
          'target_id' => $top_container->id(),
          'target_revision_id' => $top_container->getRevisionId(),
        ],
      ],
    ]);
    $deep_node->save();

    // Test the deepest paragraph shows full lineage.
    $url = Url::fromRoute('entity.paragraph.canonical', ['paragraph' => $deep_text->id()])->toString();
    $this->drupalGet($url);

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Deep Nested Node');

    // Should show multiple levels of container paragraphs in the lineage.
    $container_links = $this->getSession()
      ->getPage()
      ->findAll('css',
        'a[href*="/admin/content/paragraph-lineage/paragraph/"]');
    $this->assertGreaterThanOrEqual(2, count($container_links),
      'Should have links to multiple container paragraphs in lineage');
  }

}
