<?php

namespace Drupal\Tests\paragraph_lineage\Kernel;

use Drupal\Core\Url;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\KernelTests\KernelTestBase;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;

/**
 * Tests the theme hook functionality for paragraph lineage.
 *
 * @group paragraph_lineage
 */
class ThemeHookTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'paragraphs',
    'paragraph_lineage',
    'field',
    'text',
    'file',
    'entity_reference_revisions',
  ];

  /**
   * Test paragraph.
   *
   * @var \Drupal\paragraphs\Entity\Paragraph
   */
  protected Paragraph $paragraph;

  /**
   * Test node.
   *
   * @var \Drupal\node\Entity\Node
   */
  protected Node $node;

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('paragraph');
    $this->installConfig(['field', 'node', 'paragraphs']);

    // Create a paragraph type.
    $paragraph_type = ParagraphsType::create([
      'id' => 'test_paragraph',
      'label' => 'Test Paragraph',
    ]);
    $paragraph_type->save();

    // Create a content type.
    $node_type = NodeType::create([
      'type' => 'test_content',
      'name' => 'Test Content',
    ]);
    $node_type->save();

    // Create paragraph field on node.
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_paragraphs',
      'entity_type' => 'node',
      'type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'paragraph',
      ],
    ]);
    $field_storage->save();

    // Create the field configuration for the paragraph field.
    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'test_content',
      'settings' => [
        'handler' => 'default:paragraph',
        'handler_settings' => [
          'target_bundles' => ['test_paragraph'],
        ],
      ],
    ]);
    $field->save();

    // Create test entities.
    $this->paragraph = Paragraph::create([
      'type' => 'test_paragraph',
    ]);
    $this->paragraph->save();

    $this->node = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node',
      'field_paragraphs' => [
        [
          'target_id' => $this->paragraph->id(),
          'target_revision_id' => $this->paragraph->getRevisionId(),
        ],
      ],
    ]);
    $this->node->save();
  }

  /**
   * Tests that the paragraph_lineage theme hook is registered.
   */
  public function testThemeHookRegistration(): void {
    $theme_registry = \Drupal::service('theme.registry')->get();

    $this->assertArrayHasKey('paragraph_lineage', $theme_registry);

    $theme_hook = $theme_registry['paragraph_lineage'];
    $this->assertArrayHasKey('variables', $theme_hook);
    $this->assertArrayHasKey('paragraph', $theme_hook['variables']);
    $this->assertArrayHasKey('lineage', $theme_hook['variables']);
  }

  /**
   * Tests the theme hook implementation function exists.
   */
  public function testThemeHookImplementation(): void {
    // Test that the theme hook function can be called properly.
    $theme_hooks = paragraph_lineage_theme([], 'module', 'paragraph_lineage', '/path');

    $this->assertArrayHasKey('paragraph_lineage', $theme_hooks);

    $hook_definition = $theme_hooks['paragraph_lineage'];
    $this->assertEquals([], $hook_definition['variables']['paragraph']);
    $this->assertEquals([], $hook_definition['variables']['lineage']);
  }

  /**
   * Tests that theme variables are properly structured.
   */
  public function testThemeVariableStructure(): void {
    $theme_hooks = paragraph_lineage_theme([], 'module', 'paragraph_lineage', '/path');
    $paragraph_lineage_hook = $theme_hooks['paragraph_lineage'];

    // Verify the expected variable structure.
    $expected_variables = ['paragraph', 'lineage'];
    $actual_variables = array_keys($paragraph_lineage_hook['variables']);

    foreach ($expected_variables as $expected_var) {
      $this->assertContains($expected_var, $actual_variables);
    }
  }

  /**
   * Tests rendering with the paragraph_lineage theme hook.
   */
  public function testThemeHookRendering(): void {
    $variables = [
      'paragraph' => [
        'type' => 'test_paragraph',
        'bundle' => 'test_paragraph',
        'label' => 'Test Paragraph',
        'content' => ['#markup' => 'Test content'],
      ],
      'lineage' => [
        [
          'type' => 'node',
          'bundle' => 'test_content',
          'content' => ['#markup' => 'Parent node content'],
          'link' => ['#markup' => '<a href="/node/1">Test Node</a>'],
        ],
      ],
    ];

    $build = [
      '#theme' => 'paragraph_lineage',
      '#paragraph' => $variables['paragraph'],
      '#lineage' => $variables['lineage'],
    ];

    $renderer = \Drupal::service('renderer');
    $rendered = $renderer->renderInIsolation($build);

    // The theme should render without errors.
    $rendered_string = (string) $rendered;
    $this->assertIsString($rendered_string);
    $this->assertNotEmpty($rendered_string);
  }

  /**
   * Tests theme hook with empty variables.
   */
  public function testThemeHookWithEmptyVariables(): void {
    $build = [
      '#theme' => 'paragraph_lineage',
      '#paragraph' => [],
      '#lineage' => [],
    ];

    $renderer = \Drupal::service('renderer');
    $rendered = $renderer->renderInIsolation($build);

    // Should render without errors even with empty variables.
    $this->assertIsString((string) $rendered);
  }

  /**
   * Tests theme hook with missing variables.
   */
  public function testThemeHookWithMissingVariables(): void {
    $build = [
      '#theme' => 'paragraph_lineage',
    ];

    $renderer = \Drupal::service('renderer');
    $rendered = $renderer->renderInIsolation($build);

    // Should render without errors even with missing variables.
    $this->assertIsString((string) $rendered);
  }

  /**
   * Tests theme hook with complex lineage data.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testThemeHookWithComplexLineage(): void {
    // Create a more complex lineage structure.
    $nested_paragraph = Paragraph::create([
      'type' => 'test_paragraph',
    ]);
    $nested_paragraph->save();

    $container_paragraph = Paragraph::create([
      'type' => 'test_paragraph',
    ]);
    $container_paragraph->save();

    $complex_lineage = [
      [
        'type' => 'node',
        'bundle' => 'test_content',
        'content' => ['#markup' => 'Root node content'],
        'link' => ['#markup' => '<a href="/node/1">Root Node</a>'],
      ],
      [
        'type' => 'paragraph',
        'bundle' => 'test_paragraph',
        'content' => ['#markup' => 'Container paragraph'],
        'link' => ['#markup' => '<a href="/admin/content/paragraph-lineage/paragraph/2">Container</a>'],
      ],
      [
        'type' => 'paragraph',
        'bundle' => 'test_paragraph',
        'content' => ['#markup' => 'Nested paragraph'],
        'link' => ['#markup' => '<a href="/admin/content/paragraph-lineage/paragraph/3">Nested</a>'],
      ],
    ];

    $build = [
      '#theme' => 'paragraph_lineage',
      '#paragraph' => [
        'type' => 'paragraph',
        'bundle' => 'test_paragraph',
        'label' => 'Deeply Nested Paragraph',
        'content' => ['#markup' => 'Deep content'],
      ],
      '#lineage' => $complex_lineage,
    ];

    $renderer = \Drupal::service('renderer');
    $rendered = $renderer->renderInIsolation($build);

    $this->assertIsString((string) $rendered);
    $this->assertNotEmpty((string) $rendered);
  }

  /**
   * Tests that theme hook can handle render arrays in content.
   */
  public function testThemeHookWithRenderArrays(): void {
    $build = [
      '#theme' => 'paragraph_lineage',
      '#paragraph' => [
        'type' => 'paragraph',
        'bundle' => 'test_paragraph',
        'label' => 'Test Paragraph',
        'content' => [
          '#type' => 'markup',
          '#markup' => '<div class="paragraph-content">Rendered content</div>',
        ],
      ],
      '#lineage' => [
        [
          'type' => 'node',
          'bundle' => 'test_content',
          'content' => [
            '#type' => 'markup',
            '#markup' => '<div class="node-content">Parent content</div>',
          ],
          'link' => [
            '#type' => 'link',
            '#title' => 'Parent Node',
            '#url' => Url::fromRoute('<front>'),
          ],
        ],
      ],
    ];

    $renderer = \Drupal::service('renderer');
    $rendered = $renderer->renderInIsolation($build);

    $this->assertIsString((string) $rendered);
    $this->assertNotEmpty((string) $rendered);
  }

  /**
   * Tests theme template file exists.
   */
  public function testThemeTemplateExists(): void {
    $template_path = \Drupal::service('extension.list.module')->getPath('paragraph_lineage') . '/templates/paragraph-lineage.html.twig';

    $this->assertTrue(file_exists($template_path), 'Theme template file should exist');

    // Verify the template file has content.
    $template_content = file_get_contents($template_path);
    $this->assertNotEmpty($template_content);

    // Check for expected Twig variables in the template.
    $this->assertStringContainsString('paragraph', $template_content);
    $this->assertStringContainsString('lineage', $template_content);
  }

  /**
   * Tests theme preprocess function (if it exists).
   */
  public function testThemePreprocessFunction(): void {
    // Check if a preprocess function exists for our theme hook.
    $preprocess_functions = [];
    \Drupal::moduleHandler()->alter('theme_preprocess_paragraph_lineage', $preprocess_functions);

    // If preprocess functions exist, they should be callable.
    if (!empty($preprocess_functions)) {
      foreach ($preprocess_functions as $function) {
        $this->assertTrue(is_callable($function), "Preprocess function should be callable");
      }
    }
    else {
      // It's OK if no preprocess functions exist.
      $this->assertTrue(TRUE, 'No preprocess functions defined (acceptable)');
    }
  }

}
