<?php

namespace Drupal\Tests\paragraph_lineage\Kernel;

use Drupal\file\Entity\File;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\Tests\views\Kernel\ViewsKernelTestBase;
use Drupal\views\Views;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;

/**
 * Tests the Views integration for paragraph lineage.
 *
 * @group paragraph_lineage
 */
class ViewsIntegrationTest extends ViewsKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'paragraphs',
    'paragraph_lineage',
    'field',
    'text',
    'file',
    'entity_reference_revisions',
    'views',
  ];

  /**
   * Views used by this test.
   *
   * @var array
   */
  public static $testViews = ['test_paragraph_lineage_file_usage_paragraph_link'];

  /**
   * Test paragraph.
   *
   * @var \Drupal\paragraphs\Entity\Paragraph
   */
  protected Paragraph $paragraph;

  /**
   * Test node.
   *
   * @var \Drupal\node\Entity\Node
   */
  protected Node $node;

  /**
   * Test file.
   *
   * @var \Drupal\file\Entity\File
   */
  protected File $file;

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function setUp($import_test_views = TRUE): void {
    parent::setUp($import_test_views);

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('paragraph');
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);
    $this->installConfig(['field', 'node', 'paragraphs', 'paragraph_lineage']);

    // Create a paragraph type.
    $paragraph_type = ParagraphsType::create([
      'id' => 'test_paragraph',
      'label' => 'Test Paragraph',
    ]);
    $paragraph_type->save();

    // Create a content type.
    $node_type = NodeType::create([
      'type' => 'test_content',
      'name' => 'Test Content',
    ]);
    $node_type->save();

    // Create paragraph field on node.
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_paragraphs',
      'entity_type' => 'node',
      'type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'paragraph',
      ],
    ]);
    $field_storage->save();

    // Create the field configuration for the paragraph field.
    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'test_content',
      'settings' => [
        'handler' => 'default:paragraph',
        'handler_settings' => [
          'target_bundles' => ['test_paragraph'],
        ],
      ],
    ]);
    $field->save();

    // Create test entities.
    $this->paragraph = Paragraph::create([
      'type' => 'test_paragraph',
    ]);
    $this->paragraph->save();

    $this->node = Node::create([
      'type' => 'test_content',
      'title' => 'Test Node',
      'field_paragraphs' => [
        [
          'target_id' => $this->paragraph->id(),
          'target_revision_id' => $this->paragraph->getRevisionId(),
        ],
      ],
    ]);
    $this->node->save();

    // Create a test file.
    $this->file = File::create([
      'filename' => 'test.txt',
      'uri' => 'public://test.txt',
      'filemime' => 'text/plain',
      'status' => 1,
    ]);
    $this->file->save();

    // Add file usage entry.
    \Drupal::service('file.usage')->add($this->file, 'paragraph', 'paragraph', $this->paragraph->id());
  }

  /**
   * Tests that views data is properly altered for file usage.
   */
  public function testViewsDataAlter(): void {
    $data = Views::viewsData()->get('file_usage');

    // Check that our custom relationship was added.
    $this->assertArrayHasKey('file_to_paragraph', $data);
    $this->assertEquals('File Usage Paragraph Link', $data['file_to_paragraph']['title']);
    $this->assertEquals('standard', $data['file_to_paragraph']['relationship']['id']);

    // Check paragraph to file relationship.
    $this->assertArrayHasKey('paragraph_to_file', $data);
    $this->assertEquals('File', $data['paragraph_to_file']['title']);
    $this->assertEquals('standard', $data['paragraph_to_file']['relationship']['id']);
  }

  /**
   * Tests that the file usage paragraph link field plugin exists.
   */
  public function testFileUsageParagraphLinkFieldPlugin(): void {
    $plugin_manager = \Drupal::service('plugin.manager.views.field');
    $plugins = $plugin_manager->getDefinitions();

    // Check if our custom field plugin is available.
    $this->assertArrayHasKey('paragraph_lineage_file_usage_paragraph_link', $plugins);

    $plugin_definition = $plugins['paragraph_lineage_file_usage_paragraph_link'];
    // The plugin should have some definition properties.
    $this->assertIsArray($plugin_definition);
    $this->assertArrayHasKey('class', $plugin_definition);
  }

  /**
   * Tests the file usage paragraph link field rendering.
   */
  public function testFileUsageParagraphLinkFieldRendering(): void {
    $plugin_manager = \Drupal::service('plugin.manager.views.field');

    // Create the field plugin.
    $field_plugin = $plugin_manager->createInstance('paragraph_lineage_file_usage_paragraph_link', [
      'field' => 'paragraph_lineage_file_usage_paragraph_link',
      'table' => 'file_usage',
    ]);

    $this->assertInstanceOf('Drupal\paragraph_lineage\Plugin\views\field\FileUsageParagraphLink', $field_plugin);
  }

  /**
   * Tests file usage relationships work correctly.
   */
  public function testFileUsageRelationships(): void {
    // Query file usage table directly to verify our data setup.
    $query = \Drupal::database()->select('file_usage', 'fu')
      ->fields('fu')
      ->condition('fu.type', 'paragraph')
      ->condition('fu.id', $this->paragraph->id());

    $result = $query->execute()->fetchAssoc();

    $this->assertNotEmpty($result);
    $this->assertEquals($this->file->id(), $result['fid']);
    $this->assertEquals('paragraph', $result['type']);
    $this->assertEquals($this->paragraph->id(), $result['id']);
  }

  /**
   * Tests that views data contains proper join configuration.
   */
  public function testViewsDataJoinConfiguration(): void {
    $data = Views::viewsData()->get('file_usage');

    $file_to_paragraph = $data['file_to_paragraph'];
    $this->assertEquals('paragraphs_item_field_data', $file_to_paragraph['relationship']['base']);
    $this->assertEquals('id', $file_to_paragraph['relationship']['base field']);
    $this->assertEquals('id', $file_to_paragraph['relationship']['relationship field']);

    // Check the extra condition for paragraph type.
    $this->assertArrayHasKey('extra', $file_to_paragraph['relationship']);
    $this->assertEquals('paragraph', $file_to_paragraph['relationship']['extra'][0]['value']);
  }

  /**
   * Tests paragraph to file relationship configuration.
   */
  public function testParagraphToFileRelationship(): void {
    $data = Views::viewsData()->get('file_usage');

    $paragraph_to_file = $data['paragraph_to_file'];
    $this->assertEquals('file_managed', $paragraph_to_file['relationship']['base']);
    $this->assertEquals('fid', $paragraph_to_file['relationship']['base field']);
    $this->assertEquals('fid', $paragraph_to_file['relationship']['relationship field']);

    // Check skip base configuration.
    $expected_skip_base = [
      'file_managed',
      'users_field_data',
      'node_field_data',
      'comment_field_data',
      'taxonomy_term_field_data',
    ];
    $this->assertEquals($expected_skip_base, $paragraph_to_file['skip base']);
  }

  /**
   * Tests that the admin paragraph lineage view exists and works.
   */
  public function testAdminParagraphLineageView(): void {
    $view = Views::getView('admin_paragraph_lineage');

    if ($view) {
      $this->assertTrue($view->storage->status());

      // Execute the view to ensure it doesn't have configuration errors.
      $view->execute();

      // The view should be able to run without errors.
      $this->assertIsArray($view->result, 'View should execute without errors');
    }
    else {
      // If the view doesn't exist, that's also acceptable for testing.
      $this->markTestSkipped('Admin paragraph lineage view not installed');
    }
  }

  /**
   * Tests that relationships can be used in a view.
   */
  public function testViewsRelationshipUsage(): void {
    // Create a simple view programmatically to test the relationship.
    $view_config = [
      'id' => 'test_file_paragraph_relation',
      'label' => 'Test File Paragraph Relation',
      'base_table' => 'file_usage',
      'display' => [
        'default' => [
          'id' => 'default',
          'display_plugin' => 'default',
          'display_title' => 'Default',
          'display_options' => [
            'relationships' => [
              'file_to_paragraph' => [
                'id' => 'file_to_paragraph',
                'table' => 'file_usage',
                'field' => 'file_to_paragraph',
                'plugin_id' => 'standard',
              ],
            ],
            'fields' => [
              'id' => [
                'id' => 'id',
                'table' => 'file_usage',
                'field' => 'id',
                'plugin_id' => 'numeric',
              ],
            ],
          ],
        ],
      ],
    ];
    /** @var \Drupal\views\ViewEntityInterface $view */
    $view = \Drupal::entityTypeManager()
      ->getStorage('view')
      ->create($view_config);

    $executable = Views::executableFactory()->get($view);
    $executable->execute();

    // Should execute without errors.
    $this->assertIsArray($executable->result);
  }

  /**
   * Tests views integration with multiple file usage entries.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function testMultipleFileUsageEntries(): void {
    // Create additional test data.
    $second_paragraph = Paragraph::create([
      'type' => 'test_paragraph',
    ]);
    $second_paragraph->save();

    $second_file = File::create([
      'filename' => 'test2.txt',
      'uri' => 'public://test2.txt',
      'filemime' => 'text/plain',
      'status' => 1,
    ]);
    $second_file->save();

    // Add multiple file usage entries.
    $file_usage = \Drupal::service('file.usage');
    $file_usage->add($second_file, 'paragraph', 'paragraph', $second_paragraph->id());
    $file_usage->add($this->file, 'paragraph', 'paragraph', $second_paragraph->id());

    // Query to verify multiple relationships.
    $query = \Drupal::database()->select('file_usage', 'fu')
      ->fields('fu')
      ->condition('fu.type', 'paragraph');

    $results = $query->execute()->fetchAll();

    // Should have multiple file usage entries for paragraphs.
    $paragraph_entries = array_filter($results, function ($row) {
      return $row->type === 'paragraph';
    });

    $this->assertGreaterThanOrEqual(2, count($paragraph_entries));
  }

}
