<?php

declare(strict_types=1);

namespace Drupal\Tests\paragraph_lineage\Unit\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\paragraph_lineage\Form\ParagraphLineageSettingsForm;
use Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tests the ParagraphLineageSettingsForm.
 *
 * @group paragraph_lineage
 * @coversDefaultClass \Drupal\paragraph_lineage\Form\ParagraphLineageSettingsForm
 */
class ParagraphLineageSettingsFormTest extends UnitTestCase {

  /**
   * The configuration factory mock.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface&\PHPUnit\Framework\MockObject\MockObject
   */
  protected ConfigFactoryInterface&MockObject $configFactory;

  /**
   * The typed config manager mock.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface&\PHPUnit\Framework\MockObject\MockObject
   */
  protected TypedConfigManagerInterface&MockObject $typedConfigManager;

  /**
   * The settings service mock.
   *
   * @var \Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface&\PHPUnit\Framework\MockObject\MockObject
   */
  protected ParagraphLineageSettingsServiceInterface&MockObject $settingsService;

  /**
   * The form under test.
   *
   * @var \Drupal\paragraph_lineage\Form\ParagraphLineageSettingsForm
   */
  protected ParagraphLineageSettingsForm $form;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->typedConfigManager = $this->createMock(TypedConfigManagerInterface::class);
    $this->settingsService = $this->createMock(ParagraphLineageSettingsServiceInterface::class);

    // Create the actual form instance since it's final.
    $this->form = new ParagraphLineageSettingsForm(
      $this->configFactory,
      $this->typedConfigManager,
      $this->settingsService
    );

    // Mock string translation that returns a TranslatableMarkup-like object.
    $string_translation = $this->createMock(TranslationInterface::class);
    $string_translation->method('translate')
      ->willReturnCallback(function ($string, $args = []) {
        // Handle placeholder replacement for translated strings.
        if (!empty($args)) {
          $string = strtr($string, $args);
        }

        // Return a simple object that behaves like TranslatableMarkup.
        return new class($string) {
          /**
           * The string value.
           *
           * @var string
           */
          private $string;

          /**
           * Constructs a new anonymous class.
           *
           * @param string $string
           *   The string value.
           */
          public function __construct($string) {
            $this->string = $string;
          }

          /**
           * Returns the string representation.
           *
           * @return string
           *   The string value.
           */
          public function __toString() {
            return $this->string;
          }

          /**
           * Renders the string.
           *
           * @return string
           *   The string value.
           */
          public function render() {
            return $this->string;
          }

        };
      });
    $this->form->setStringTranslation($string_translation);
  }

  /**
   * Tests the form ID.
   *
   * @covers ::getFormId
   */
  public function testGetFormId(): void {
    $this->assertEquals('paragraph_lineage_settings', $this->form->getFormId());
  }

  /**
   * Tests getEditableConfigNames.
   *
   * @covers ::getEditableConfigNames
   */
  public function testGetEditableConfigNames(): void {
    $method = new \ReflectionMethod($this->form, 'getEditableConfigNames');
    $method->setAccessible(TRUE);
    $result = $method->invoke($this->form);
    $this->assertEquals(['paragraph_lineage.settings'], $result);
  }

  /**
   * Tests buildForm structure and default values.
   *
   * @covers ::buildForm
   */
  public function testBuildForm(): void {
    $this->settingsService->expects($this->once())
      ->method('getPreferredThemeId')
      ->with(TRUE)
      ->willReturn('admin');

    $this->settingsService->expects($this->once())
      ->method('isPreferredThemeOverridden')
      ->willReturn(FALSE);

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $this->form->buildForm($form, $form_state);

    $this->assertArrayHasKey('preferred_theme', $result);
    $this->assertEquals('radios', $result['preferred_theme']['#type']);
    $this->assertArrayHasKey('#title', $result['preferred_theme']);
    $this->assertArrayHasKey('#description', $result['preferred_theme']);
    $this->assertEquals('admin', $result['preferred_theme']['#default_value']);
    $this->assertArrayHasKey('default', $result['preferred_theme']['#options']);
    $this->assertArrayHasKey('admin', $result['preferred_theme']['#options']);
  }

  /**
   * Tests submitForm calls the settings service.
   *
   * @covers ::submitForm
   */
  public function testSubmitForm(): void {
    // Mock messenger service and container to handle parent::submitForm()
    $messenger = $this->createMock(MessengerInterface::class);
    $messenger->expects($this->once())
      ->method('addStatus');

    $container = $this->createMock(ContainerInterface::class);
    $container->method('get')
      ->with('messenger')
      ->willReturn($messenger);

    \Drupal::setContainer($container);

    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('getValue')
      ->with('preferred_theme')
      ->willReturn('default');

    $this->settingsService->expects($this->once())
      ->method('setPreferredThemeId')
      ->with('default');

    $form = [];
    $this->form->submitForm($form, $form_state);
  }

  /**
   * Tests buildForm when config is overridden in settings.php.
   *
   * This test simulates the scenario where a site administrator has overridden
   * the preferred theme setting in settings.php, which makes the form field
   * read-only and displays an informational message.
   *
   * Steps:
   * 1. Set up container with messenger service for override warning
   * 2. Mock the service to return TRUE for isPreferredThemeOverridden()
   * 3. Mock getPreferredThemeId calls (3 times: form default, warning, message)
   * 4. Build the form and verify it's disabled with the correct message.
   *
   * How overrides work:
   * - The service compares editable config vs immutable config
   * - If they differ, it means settings.php has an override.
   * - The form shows the editable value as default but displays the override.
   * - The field is disabled to prevent confusion since changes won't work.
   *
   * @covers ::buildForm
   */
  public function testBuildFormWithOverriddenConfig(): void {
    // Step 1: Set up container with messenger service for override warning.
    $messenger = $this->createMock(MessengerInterface::class);
    $messenger->expects($this->once())
      ->method('addWarning');

    $container = $this->createMock(ContainerInterface::class);
    $container->method('get')
      ->with('messenger')
      ->willReturn($messenger);

    \Drupal::setContainer($container);

    // Step 2: Mock the service to indicate the config is overridden.
    $this->settingsService->expects($this->once())
      ->method('isPreferredThemeOverridden')
      ->willReturn(TRUE);

    // Step 3: Mock getPreferredThemeId calls - the form calls this three times:
    // - First with TRUE for the form's default value (editable config).
    // - Second with no parameter for the warning message (immutable config).
    // - Third with no parameter for the override message (immutable config).
    $this->settingsService->expects($this->exactly(3))
      ->method('getPreferredThemeId')
      ->willReturnCallback(function ($editable = FALSE) {
        return $editable ? 'admin' : 'custom';
      });

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    // Step 4: Build the form.
    $result = $this->form->buildForm($form, $form_state);

    // Step 5: Verify the form structure and override behavior.
    $this->assertArrayHasKey('preferred_theme', $result);
    $this->assertEquals('radios', $result['preferred_theme']['#type']);
    $this->assertEquals('admin', $result['preferred_theme']['#default_value']);
    $this->assertTrue($result['preferred_theme']['#disabled']);

    // Verify the override message is displayed.
    $this->assertArrayHasKey('#description', $result['preferred_theme']);
    // Verify that the description is TranslatableMarkup.
    $description = $result['preferred_theme']['#description'];
    $this->assertInstanceOf(TranslatableMarkup::class, $description);
    // Get the string value in the TranslatableMarkup $description.
    $this->assertStringContainsString('This setting is overridden as', $description->getUntranslatedString());
    // Verify the placeholder is replaced with the overridden setting value.
    $this->assertStringContainsString('custom', $description->getArguments()['@setting']);
  }

  /**
   * Tests buildForm when config is not overridden.
   *
   * @covers ::buildForm
   */
  public function testBuildFormWithoutOverriddenConfig(): void {
    $this->settingsService->expects($this->once())
      ->method('getPreferredThemeId')
      ->with(TRUE)
      ->willReturn('default');

    $this->settingsService->expects($this->once())
      ->method('isPreferredThemeOverridden')
      ->willReturn(FALSE);

    $form_state = $this->createMock(FormStateInterface::class);
    $form = [];

    $result = $this->form->buildForm($form, $form_state);

    $this->assertArrayHasKey('preferred_theme', $result);
    $this->assertEquals('radios', $result['preferred_theme']['#type']);
    $this->assertEquals('default', $result['preferred_theme']['#default_value']);
    $this->assertArrayNotHasKey('#disabled', $result['preferred_theme']);
    $this->assertStringNotContainsString('overridden in settings.php', (string) $result['preferred_theme']['#description']);
  }

}
