<?php

declare(strict_types=1);

namespace Drupal\Tests\paragraph_lineage\Unit;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\paragraph_lineage\ParagraphLineageSettingsService;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the ParagraphLineageSettingsService.
 *
 * @group paragraph_lineage
 * @coversDefaultClass \Drupal\paragraph_lineage\ParagraphLineageSettingsService
 */
class ParagraphLineageSettingsServiceTest extends UnitTestCase {

  /**
   * The configuration factory mock.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface&\PHPUnit\Framework\MockObject\MockObject
   */
  protected ConfigFactoryInterface&MockObject $configFactory;


  /**
   * The service under test.
   *
   * @var \Drupal\paragraph_lineage\ParagraphLineageSettingsService
   */
  protected ParagraphLineageSettingsService $settingsService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Step 1: Create a mock of the ConfigFactory service.
    // This will be used to simulate Drupal's configuration system.
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);

    // Step 2: Create the settings service with the mocked dependency.
    // This is the actual service we're testing.
    $this->settingsService = new ParagraphLineageSettingsService(
      $this->configFactory
    );
  }

  /**
   * Tests getPreferredThemeId with a configured theme.
   *
   * @covers ::getPreferredThemeId
   */
  public function testGetPreferredThemeIdWithConfiguredTheme(): void {
    // Step 1: Create a mock Config object that will represent the loaded
    // configuration.
    // This simulates what Drupal's config system returns when loading a
    // config file.
    $config = $this->createMock(Config::class);

    // Step 2: Set up the config mock to return 'default' when
    // get('preferred_theme') is called.
    // This simulates the configuration file having preferred_theme: default.
    $config->expects($this->once())
      ->method('get')
      ->with('preferred_theme')
      ->willReturn('default');

    // Step 3: Set up the config factory mock to return our config mock.
    // The service calls get(), so we need to mock that method.
    $this->configFactory->expects($this->once())
      ->method('get')
      ->with(ParagraphLineageSettingsService::CONFIG_NAME)
      ->willReturn($config);

    // Step 4: Call the method we're testing.
    $result = $this->settingsService->getPreferredThemeId();

    // Step 5: Assert that the result matches what we expect.
    $this->assertEquals('default', $result);
  }

  /**
   * Tests getPreferredThemeId falls back to admin theme.
   *
   * @covers ::getPreferredThemeId
   */
  public function testGetPreferredThemeIdFallsBackToAdminTheme(): void {
    // Step 1: Create a mock Config object.
    $settingsConfig = $this->createMock(Config::class);

    // Step 2: Set up the config mock to return NULL when get('preferred_theme')
    // is called.
    // This simulates the configuration file either not having the key or having
    // a null value.
    $settingsConfig->expects($this->once())
      ->method('get')
      ->with('preferred_theme')
      ->willReturn(NULL);

    // Step 3: Set up the config factory mock to return our config mock.
    // Use get() to match what the actual service calls.
    $this->configFactory->expects($this->once())
      ->method('get')
      ->with(ParagraphLineageSettingsService::CONFIG_NAME)
      ->willReturn($settingsConfig);

    // Step 4: Call the method we're testing.
    $result = $this->settingsService->getPreferredThemeId();

    // Step 5: Assert that the fallback value 'admin' is returned.
    $this->assertEquals('admin', $result);
  }

  /**
   * Tests getPreferredThemeId with editable parameter.
   *
   * @covers ::getPreferredThemeId
   */
  public function testGetPreferredThemeIdWithEditableParameter(): void {
    // Step 1: Create a mock Config object.
    $config = $this->createMock(Config::class);

    // Step 2: Set up the config mock to return 'custom' when
    // get('preferred_theme') is called.
    $config->expects($this->once())
      ->method('get')
      ->with('preferred_theme')
      ->willReturn('custom');

    // Step 3: Set up the config factory mock to return our config mock.
    // When editable=TRUE, it should call getEditable().
    $this->configFactory->expects($this->once())
      ->method('getEditable')
      ->with(ParagraphLineageSettingsService::CONFIG_NAME)
      ->willReturn($config);

    // Step 4: Call the method we're testing with editable=TRUE.
    $result = $this->settingsService->getPreferredThemeId(TRUE);

    // Step 5: Assert that the result matches what we expect.
    $this->assertEquals('custom', $result);
  }

  /**
   * Tests setPreferredThemeId saves the theme.
   *
   * @covers ::setPreferredThemeId
   */
  public function testSetPreferredThemeId(): void {
    // Step 1: Create a mock Config object that will represent the editable
    // configuration.
    $config = $this->createMock(Config::class);

    // Step 2: Set up the config mock to expect set() to be called with specific
    // parameters.
    // This simulates setting the preferred_theme configuration value to
    // 'default'.
    $config->expects($this->once())
      ->method('set')
      ->with('preferred_theme', 'default')
      ->willReturnSelf();

    // Step 3: Set up the config mock to expect save() to be called.
    // This simulates persisting the configuration changes.
    $config->expects($this->once())
      ->method('save')
      ->willReturnSelf();

    // Step 4: Set up the config factory mock to return our config mock.
    // The service calls getEditable() to get a config object that can be
    // modified.
    $this->configFactory->expects($this->once())
      ->method('getEditable')
      ->with(ParagraphLineageSettingsService::CONFIG_NAME)
      ->willReturn($config);

    // Step 5: Call the method we're testing.
    // This should trigger the set() and save() calls on the config object.
    $this->settingsService->setPreferredThemeId('default');
  }

  /**
   * Tests getPreferredThemeId falls back to admin when no admin theme.
   *
   * @covers ::getPreferredThemeId
   */
  public function testGetPreferredThemeIdFallsBackToAdmin(): void {
    // Step 1: Create a mock Config object.
    $settingsConfig = $this->createMock(Config::class);

    // Step 2: Set up the config mock to return NULL when get('preferred_theme')
    // is called.
    // This tests the same fallback behavior as the previous test.
    $settingsConfig->expects($this->once())
      ->method('get')
      ->with('preferred_theme')
      ->willReturn(NULL);

    // Step 3: Set up the config factory mock to return our config mock.
    // Use get() to match what the actual service calls.
    $this->configFactory->expects($this->once())
      ->method('get')
      ->with(ParagraphLineageSettingsService::CONFIG_NAME)
      ->willReturn($settingsConfig);

    // Step 4: Call the method we're testing.
    $result = $this->settingsService->getPreferredThemeId();

    // Step 5: Assert that the fallback value 'admin' is returned.
    $this->assertEquals('admin', $result);
  }

}
