<?php

declare(strict_types=1);

namespace Drupal\Tests\paragraph_lineage\Unit\Theme;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface;
use Drupal\paragraph_lineage\Theme\ParagraphLineageThemeNegotiator;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the ParagraphLineageThemeNegotiator.
 *
 * @group paragraph_lineage
 * @coversDefaultClass \Drupal\paragraph_lineage\Theme\ParagraphLineageThemeNegotiator
 */
class ParagraphLineageThemeNegotiatorTest extends UnitTestCase {

  /**
   * The route match mock.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface&\PHPUnit\Framework\MockObject\MockObject
   */
  protected RouteMatchInterface&MockObject $routeMatch;

  /**
   * The settings service mock.
   *
   * @var \Drupal\paragraph_lineage\ParagraphLineageSettingsServiceInterface&\PHPUnit\Framework\MockObject\MockObject
   */
  protected ParagraphLineageSettingsServiceInterface&MockObject $settingsService;

  /**
   * The config factory mock.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface&\PHPUnit\Framework\MockObject\MockObject
   */
  protected ConfigFactoryInterface&MockObject $configFactory;

  /**
   * The theme negotiator under test.
   *
   * @var \Drupal\paragraph_lineage\Theme\ParagraphLineageThemeNegotiator
   */
  protected ParagraphLineageThemeNegotiator $themeNegotiator;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->routeMatch = $this->createMock(RouteMatchInterface::class);
    $this->settingsService = $this->createMock(ParagraphLineageSettingsServiceInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);

    $this->themeNegotiator = new ParagraphLineageThemeNegotiator(
      $this->routeMatch,
      $this->settingsService,
      $this->configFactory
    );
  }

  /**
   * Tests applies() returns true for paragraph canonical route.
   *
   * @covers ::applies
   */
  public function testAppliesReturnsTrueForParagraphRoute(): void {
    $route_match = $this->createMock(RouteMatchInterface::class);
    $route_match->expects($this->once())
      ->method('getRouteName')
      ->willReturn('entity.paragraph.canonical');

    $result = $this->themeNegotiator->applies($route_match);
    $this->assertTrue($result);
  }

  /**
   * Tests applies() returns false for other routes.
   *
   * @covers ::applies
   */
  public function testAppliesReturnsFalseForOtherRoutes(): void {
    $route_match = $this->createMock(RouteMatchInterface::class);
    $route_match->expects($this->once())
      ->method('getRouteName')
      ->willReturn('some.other.route');

    $result = $this->themeNegotiator->applies($route_match);
    $this->assertFalse($result);
  }

  /**
   * Tests determineActiveTheme with 'default' preference.
   *
   * @covers ::determineActiveTheme
   */
  public function testDetermineActiveThemeWithDefaultPreference(): void {
    $route_match = $this->createMock(RouteMatchInterface::class);

    $this->settingsService->expects($this->once())
      ->method('getPreferredThemeId')
      ->with(FALSE)
      ->willReturn('default');

    $config = $this->createMock(Config::class);
    $config->expects($this->once())
      ->method('get')
      ->with('default')
      ->willReturn('olivero');

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('system.theme')
      ->willReturn($config);

    $result = $this->themeNegotiator->determineActiveTheme($route_match);
    $this->assertEquals('olivero', $result);
  }

  /**
   * Tests determineActiveTheme with 'admin' preference.
   *
   * @covers ::determineActiveTheme
   */
  public function testDetermineActiveThemeWithAdminPreference(): void {
    $route_match = $this->createMock(RouteMatchInterface::class);

    $this->settingsService->expects($this->once())
      ->method('getPreferredThemeId')
      ->with(FALSE)
      ->willReturn('admin');

    $config = $this->createMock(Config::class);
    $config->expects($this->once())
      ->method('get')
      ->with('admin')
      ->willReturn('gin');

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('system.theme')
      ->willReturn($config);

    $result = $this->themeNegotiator->determineActiveTheme($route_match);
    $this->assertEquals('gin', $result);
  }

  /**
   * Tests determineActiveTheme with custom theme preference.
   *
   * @covers ::determineActiveTheme
   */
  public function testDetermineActiveThemeWithCustomPreference(): void {
    $route_match = $this->createMock(RouteMatchInterface::class);

    $this->settingsService->expects($this->once())
      ->method('getPreferredThemeId')
      ->with(FALSE)
      ->willReturn('custom_theme');

    // Config factory should not be called for custom themes.
    $this->configFactory->expects($this->never())
      ->method('get');

    $result = $this->themeNegotiator->determineActiveTheme($route_match);
    $this->assertEquals('custom_theme', $result);
  }

  /**
   * Tests determineActiveTheme with null system theme config.
   *
   * @covers ::determineActiveTheme
   */
  public function testDetermineActiveThemeWithNullSystemConfig(): void {
    $route_match = $this->createMock(RouteMatchInterface::class);

    $this->settingsService->expects($this->once())
      ->method('getPreferredThemeId')
      ->with(FALSE)
      ->willReturn('admin');

    $config = $this->createMock(Config::class);
    $config->expects($this->once())
      ->method('get')
      ->with('admin')
      ->willReturn(NULL);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('system.theme')
      ->willReturn($config);

    $result = $this->themeNegotiator->determineActiveTheme($route_match);
    $this->assertNull($result);
  }

}
