<?php

namespace Drupal\paragraph_sdc\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Url;
use Drupal\paragraphs\ParagraphInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Controller for paragraph edit and view routes.
 */
class ParagraphEditController extends ControllerBase {

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a ParagraphEditController object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(ModuleHandlerInterface $module_handler) {
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler')
    );
  }

  /**
   * Redirects to the appropriate edit form for a paragraph.
   *
   * @param \Drupal\paragraphs\ParagraphInterface $paragraph
   *   The paragraph entity.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect to the edit form.
   */
  public function edit(ParagraphInterface $paragraph) {
    // Get the root parent entity by traversing up the hierarchy.
    $root_parent = $this->getRootParent($paragraph);

    if (!$root_parent) {
      $this->messenger()->addError($this->t('Unable to find parent entity for this paragraph.'));
      return $this->redirect('<front>');
    }

    // If paragraphs_edit module is enabled, use its route.
    if ($this->moduleHandler->moduleExists('paragraphs_edit')) {
      $url = Url::fromRoute('paragraphs_edit.edit_form', [
        'root_parent_type' => $root_parent->getEntityTypeId(),
        'root_parent' => $root_parent->id(),
        'paragraph' => $paragraph->id(),
      ]);
      return new RedirectResponse($url->toString());
    }

    // Otherwise, redirect to parent entity edit form.
    // You could enhance this to open the specific paragraph field.
    $url = $root_parent->toUrl('edit-form');
    return new RedirectResponse($url->toString());
  }

  /**
   * Renders a paragraph entity.
   *
   * @param \Drupal\paragraphs\ParagraphInterface $paragraph
   *   The paragraph entity.
   *
   * @return array
   *   Render array for the paragraph.
   */
  public function view(ParagraphInterface $paragraph) {
    $view_builder = $this->entityTypeManager()->getViewBuilder('paragraph');
    return $view_builder->view($paragraph, 'default');
  }

  /**
   * Gets the root parent of a paragraph by traversing up the hierarchy.
   *
   * @param \Drupal\paragraphs\ParagraphInterface $paragraph
   *   The paragraph entity.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface|null
   *   The root parent entity, or NULL if not found.
   */
  protected function getRootParent(ParagraphInterface $paragraph) {
    $root_parent = $paragraph->getParentEntity();

    // Keep traversing up while the parent is still a paragraph.
    while ($root_parent instanceof ParagraphInterface) {
      $root_parent = $root_parent->getParentEntity();
    }

    return $root_parent;
  }

}
