<?php

namespace Drupal\paragraph_sdc;

use Drupal\Core\Url;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

/**
 * Twig extension for generating entity edit URLs.
 */
class EntityEditTwigExtension extends AbstractExtension {

  /**
   * {@inheritdoc}
   */
  public function getFunctions() {
    return [
      new TwigFunction('entity_edit_url', [$this, 'getEntityEditUrl']),
    ];
  }

  /**
   * Generates an edit URL for any entity type.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param int|string $entity_id
   *   The entity ID.
   *
   * @return string
   *   The edit URL, or empty string if entity type doesn't support editing.
   */
  public function getEntityEditUrl($entity_type, $entity_id) {
    // Handle each entity type with its specific URL pattern.
    switch ($entity_type) {
      case 'paragraph':
        // Use our custom route for paragraphs.
        try {
          $url = Url::fromRoute('paragraph_sdc.edit', [
            'paragraph' => $entity_id,
          ]);
          return $url->toString();
        }
        catch (\Exception $e) {
          return '/paragraph/' . $entity_id . '/edit';
        }

      case 'node':
        // Pattern: /node/{id}/edit
        return '/node/' . $entity_id . '/edit';

      case 'user':
        // Pattern: /user/{id}/edit
        return '/user/' . $entity_id . '/edit';

      case 'taxonomy_term':
        // Pattern: /taxonomy/term/{id}/edit
        return '/taxonomy/term/' . $entity_id . '/edit';

      case 'comment':
        // Pattern: /comment/{id}/edit
        return '/comment/' . $entity_id . '/edit';

      case 'file':
        // Files don't have an edit form accessible to users.
        // Return empty string or home page.
        return '/';

      case 'block_content':
        // Pattern: /admin/content/block/{id}
        return '/admin/content/block/' . $entity_id;

      case 'contact_form':
        // Pattern: /admin/structure/contact/messages/{id}/edit
        return '/admin/structure/contact/messages/' . $entity_id . '/edit';

      default:
        // For unknown entity types, try the standard Drupal route pattern.
        try {
          $url = Url::fromRoute("entity.{$entity_type}.edit_form", [
            $entity_type => $entity_id,
          ]);
          return $url->toString();
        }
        catch (\Exception $e) {
          // If route doesn't exist, return a generic pattern.
          return "/{$entity_type}/{$entity_id}/edit";
        }
    }
  }

}
