<?php

namespace Drupal\paragraph_usage_dashboard\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\State\StateInterface;
use Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the paragraph usage dashboard.
 */
class DashboardController extends ControllerBase {

  /**
   * The paragraph usage collector service.
   *
   * @var \Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector
   */
  protected $usageCollector;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a DashboardController object.
   *
   * @param \Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector $usage_collector
   *   The paragraph usage collector service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    ParagraphUsageCollector $usage_collector,
    StateInterface $state
  ) {
    $this->usageCollector = $usage_collector;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('paragraph_usage_dashboard.collector'),
      $container->get('state')
    );
  }

  /**
   * Displays the paragraph usage dashboard.
   *
   * @return array
   *   A render array.
   */
  public function dashboard() {
    $selected_types = $this->state->get('paragraph_usage_dashboard.selected_types', []);
    $path_search = $this->state->get('paragraph_usage_dashboard.path_search', '');
    $name_search = $this->state->get('paragraph_usage_dashboard.name_search', '');
    $usage_data = $this->usageCollector->collectUsageData($selected_types, $path_search, $name_search);

    // Build filter form.
    $filter_form = $this->formBuilder()->getForm('Drupal\paragraph_usage_dashboard\Form\ParagraphFilterForm');

    // Get all paragraph types to check for unused ones.
    $all_paragraph_types = $this->usageCollector->getAllParagraphTypes();

    // Build table rows.
    $rows = [];
    foreach ($usage_data as $type_id => $data) {
      $icon_render = [];
      if ($data['icon']) {
        $icon_render = [
          '#theme' => 'image_style',
          '#style_name' => 'thumbnail',
          '#uri' => $data['icon']->getFileUri(),
          '#alt' => $this->t('Icon for @type', ['@type' => $data['type']->label()]),
          '#width' => 50,
          '#height' => 50,
        ];
      }

      $content_types = !empty($data['content_types']) ? implode(', ', $data['content_types']) : $this->t('None');

      // Build path links.
      $path_links = [];
      if (!empty($data['paths'])) {
        foreach ($data['paths'] as $path) {
          $path_links[] = [
            '#type' => 'link',
            '#title' => $path,
            '#url' => \Drupal\Core\Url::fromUserInput($path),
          ];
          $path_links[] = ['#markup' => ', '];
        }
        // Remove last comma.
        array_pop($path_links);
      }
      else {
        $path_links = ['#markup' => $this->t('None')];
      }

      // Check if paragraph type is unused.
      $is_unused = empty($data['entities']);

      // Link paragraph type name to backend configuration.
      $paragraph_type_name = [
        '#type' => 'container',
      ];

      $paragraph_type_name['link'] = [
        '#type' => 'link',
        '#title' => $data['type']->label(),
        '#url' => \Drupal\Core\Url::fromRoute('entity.paragraphs_type.edit_form', [
          'paragraphs_type' => $type_id,
        ]),
      ];

      // Add unused badge if applicable.
      if ($is_unused) {
        $paragraph_type_name['badge'] = [
          '#markup' => ' <span class="paragraph-unused-badge">' . $this->t('Unused') . '</span>',
        ];
      }

      // Add "View Usage Details" button.
      $detail_button = [
        '#type' => 'link',
        '#title' => $this->t('View Usage Details'),
        '#url' => \Drupal\Core\Url::fromRoute('paragraph_usage_dashboard.detail', [
          'paragraph_type' => $type_id,
        ]),
        '#attributes' => ['class' => ['button', 'button--small', 'button--primary']],
      ];

      $rows[] = [
        ['data' => $icon_render],
        ['data' => $paragraph_type_name],
        ['data' => ['#markup' => $content_types]],
        ['data' => $path_links],
        ['data' => $detail_button],
      ];
    }

    $build = [];

    $build['filter'] = $filter_form;
    $build['table'] = [
      '#theme' => 'table',
      '#header' => [
        $this->t('Icon'),
        $this->t('Paragraph Type'),
        $this->t('Used in Content Type(s)'),
        $this->t('Path Alias(es)'),
        $this->t('Usage Details'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No paragraph usage found.'),
      '#attributes' => ['class' => ['paragraph-usage-dashboard-table']],
    ];

    $build['#attached']['library'][] = 'paragraph_usage_dashboard/dashboard';

    return $build;
  }

}
