<?php

namespace Drupal\paragraph_usage_dashboard\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for detailed paragraph usage view.
 */
class ParagraphDetailController extends ControllerBase {

  /**
   * The paragraph usage collector service.
   *
   * @var \Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector
   */
  protected $usageCollector;

  /**
   * Constructs a ParagraphDetailController object.
   *
   * @param \Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector $usage_collector
   *   The paragraph usage collector service.
   */
  public function __construct(ParagraphUsageCollector $usage_collector) {
    $this->usageCollector = $usage_collector;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('paragraph_usage_dashboard.collector')
    );
  }

  /**
   * Displays detailed usage for a specific paragraph type.
   *
   * @param string $paragraph_type
   *   The paragraph type ID.
   *
   * @return array
   *   A render array.
   */
  public function detail($paragraph_type) {
    $usage_data = $this->usageCollector->getDetailedUsage($paragraph_type);

    if (empty($usage_data)) {
      return [
        '#markup' => $this->t('No usage data found for this paragraph type.'),
      ];
    }

    $paragraph_type_entity = $usage_data['type'];

    // Build detailed table.
    $rows = [];
    foreach ($usage_data['entities'] as $item) {
      $entity = $item['entity'];
      $field_name = $item['field_name'];

      $entity_type_id = $entity->getEntityTypeId();
      $bundle = $entity->bundle();
      $bundle_label = $this->usageCollector->getBundleLabel($entity_type_id, $bundle);

      $path_alias = $this->usageCollector->getEntityPathAlias($entity);

      // Build action links.
      $action_links = [];
      if ($path_alias) {
        // Preview link (opens in new tab).
        $action_links[] = [
          '#type' => 'link',
          '#title' => $this->t('Preview'),
          '#url' => \Drupal\Core\Url::fromUserInput($path_alias),
          '#attributes' => [
            'class' => ['button', 'button--small', 'button--primary'],
            'target' => '_blank',
          ],
        ];
        $action_links[] = ['#markup' => ' '];
      }

      // Edit link.
      if ($entity->hasLinkTemplate('edit-form')) {
        $action_links[] = [
          '#type' => 'link',
          '#title' => $this->t('Edit'),
          '#url' => $entity->toUrl('edit-form'),
          '#attributes' => ['class' => ['button', 'button--small']],
        ];
        $action_links[] = ['#markup' => ' '];
      }

      // Copy path button.
      if ($path_alias) {
        $action_links[] = [
          '#type' => 'html_tag',
          '#tag' => 'button',
          '#value' => $this->t('Copy Path'),
          '#attributes' => [
            'class' => ['button', 'button--small', 'copy-path-btn'],
            'data-path' => $path_alias,
            'type' => 'button',
          ],
        ];
      }

      // Build path link.
      $path_link = ['#markup' => $this->t('No alias')];
      if ($path_alias) {
        $path_link = [
          '#type' => 'link',
          '#title' => $path_alias,
          '#url' => \Drupal\Core\Url::fromUserInput($path_alias),
        ];
      }

      $rows[] = [
        ['data' => ['#markup' => $entity->label() ?: $this->t('Untitled')]],
        ['data' => ['#markup' => $bundle_label]],
        ['data' => ['#markup' => $field_name]],
        ['data' => $path_link],
        ['data' => $action_links],
      ];
    }

    $build = [];

    $build['info'] = [
      '#markup' => '<p>' . $this->t('Detailed usage information for <strong>@type</strong>', [
        '@type' => $paragraph_type_entity->label(),
      ]) . '</p>',
    ];

    $build['back_link'] = [
      '#type' => 'link',
      '#title' => $this->t('← Back to Dashboard'),
      '#url' => \Drupal\Core\Url::fromRoute('paragraph_usage_dashboard.dashboard'),
      '#attributes' => ['class' => ['button']],
      '#prefix' => '<div class="paragraph-detail-back">',
      '#suffix' => '</div>',
    ];

    $build['table'] = [
      '#theme' => 'table',
      '#header' => [
        $this->t('Entity Title'),
        $this->t('Content Type'),
        $this->t('Field Name'),
        $this->t('Path Alias'),
        $this->t('Actions'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No instances found.'),
      '#attributes' => ['class' => ['paragraph-detail-table']],
    ];

    $build['#attached']['library'][] = 'paragraph_usage_dashboard/dashboard';

    return $build;
  }

}
