<?php

namespace Drupal\paragraph_usage_dashboard\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\State\StateInterface;
use Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for filtering paragraph types in the dashboard.
 */
class ParagraphFilterForm extends FormBase {

  /**
   * The paragraph usage collector service.
   *
   * @var \Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector
   */
  protected $usageCollector;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a ParagraphFilterForm object.
   *
   * @param \Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector $usage_collector
   *   The paragraph usage collector service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    ParagraphUsageCollector $usage_collector,
    StateInterface $state
  ) {
    $this->usageCollector = $usage_collector;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('paragraph_usage_dashboard.collector'),
      $container->get('state')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'paragraph_usage_dashboard_filter_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $paragraph_types = $this->usageCollector->getAllParagraphTypes();
    $options = [];

    foreach ($paragraph_types as $type_id => $type) {
      $options[$type_id] = $type->label();
    }

    $selected_types = $this->state->get('paragraph_usage_dashboard.selected_types', []);
    $path_search = $this->state->get('paragraph_usage_dashboard.path_search', '');
    $name_search = $this->state->get('paragraph_usage_dashboard.name_search', '');

    $form['filter'] = [
      '#type' => 'details',
      '#title' => $this->t('Filter Paragraph Types'),
      '#open' => !empty($selected_types) || !empty($path_search) || !empty($name_search),
    ];

    $form['filter']['name_search'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search by Paragraph Type Name'),
      '#default_value' => $name_search,
      '#placeholder' => $this->t('e.g., Banner, Hero, Card'),
      '#description' => $this->t('Search for paragraph types by name. Partial matches supported.'),
      '#size' => 60,
    ];

    $form['filter']['path_search'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search by Path Alias'),
      '#default_value' => $path_search,
      '#placeholder' => $this->t('e.g., /chss/center or research'),
      '#description' => $this->t('Search for paragraph components used in pages with this path. Partial matches supported.'),
      '#size' => 60,
    ];

    $form['filter']['selected_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Select paragraph types to display'),
      '#options' => $options,
      '#default_value' => $selected_types,
      '#description' => $this->t('Leave all unchecked to show all paragraph types.'),
    ];

    $form['filter']['actions'] = [
      '#type' => 'actions',
    ];

    $form['filter']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Apply Filter'),
      '#button_type' => 'primary',
    ];

    $form['filter']['actions']['reset'] = [
      '#type' => 'submit',
      '#value' => $this->t('Reset Filter'),
      '#submit' => ['::resetFilter'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $selected_types = array_filter($form_state->getValue('selected_types'));
    $path_search = trim($form_state->getValue('path_search'));
    $name_search = trim($form_state->getValue('name_search'));

    $this->state->set('paragraph_usage_dashboard.selected_types', $selected_types);
    $this->state->set('paragraph_usage_dashboard.path_search', $path_search);
    $this->state->set('paragraph_usage_dashboard.name_search', $name_search);

    $this->messenger()->addStatus($this->t('Filter applied successfully.'));
  }

  /**
   * Reset filter submit handler.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function resetFilter(array &$form, FormStateInterface $form_state) {
    $this->state->delete('paragraph_usage_dashboard.selected_types');
    $this->state->delete('paragraph_usage_dashboard.path_search');
    $this->state->delete('paragraph_usage_dashboard.name_search');
    $this->messenger()->addStatus($this->t('Filter reset successfully.'));
  }

}
