<?php

namespace Drupal\paragraph_usage_dashboard\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\file\FileUsage\FileUsageInterface;

/**
 * Service to collect paragraph usage data across the site.
 */
class ParagraphUsageCollector {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The path alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $pathAliasManager;

  /**
   * The file usage service.
   *
   * @var \Drupal\file\FileUsage\FileUsageInterface
   */
  protected $fileUsage;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * Constructs a ParagraphUsageCollector object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\path_alias\AliasManagerInterface $path_alias_manager
   *   The path alias manager.
   * @param \Drupal\file\FileUsage\FileUsageInterface $file_usage
   *   The file usage service.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    AliasManagerInterface $path_alias_manager,
    FileUsageInterface $file_usage,
    EntityRepositoryInterface $entity_repository
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->pathAliasManager = $path_alias_manager;
    $this->fileUsage = $file_usage;
    $this->entityRepository = $entity_repository;
  }

  /**
   * Get all paragraph types.
   *
   * @return array
   *   Array of paragraph type entities keyed by ID.
   */
  public function getAllParagraphTypes() {
    return $this->entityTypeManager
      ->getStorage('paragraphs_type')
      ->loadMultiple();
  }

  /**
   * Get the icon file entity for a paragraph type.
   *
   * @param \Drupal\paragraphs\Entity\ParagraphsType $paragraph_type
   *   The paragraph type entity.
   *
   * @return \Drupal\file\FileInterface|null
   *   The file entity or NULL if not found.
   */
  public function getParagraphTypeIcon($paragraph_type) {
    $icon_uuid = $paragraph_type->get('icon_uuid');
    if (empty($icon_uuid)) {
      return NULL;
    }

    $files = $this->entityTypeManager
      ->getStorage('file')
      ->loadByProperties(['uuid' => $icon_uuid]);

    return $files ? reset($files) : NULL;
  }

  /**
   * Collect usage data for all paragraph types.
   *
   * @param array $selected_types
   *   Array of selected paragraph type IDs to filter by.
   * @param string $path_search
   *   Optional path search string to filter by path aliases.
   * @param string $name_search
   *   Optional name search string to filter by paragraph type name.
   *
   * @return array
   *   Array of usage data keyed by paragraph type ID.
   */
  public function collectUsageData(array $selected_types = [], $path_search = '', $name_search = '') {
    $usage_data = [];
    $paragraph_types = $this->getAllParagraphTypes();

    foreach ($paragraph_types as $type_id => $paragraph_type) {
      // Filter by selected types if provided.
      if (!empty($selected_types) && !in_array($type_id, $selected_types)) {
        continue;
      }

      // Filter by name search if provided.
      if (!empty($name_search)) {
        $label = strtolower($paragraph_type->label());
        $search = strtolower(trim($name_search));
        if (stripos($label, $search) === FALSE) {
          continue;
        }
      }

      $usage_data[$type_id] = [
        'type' => $paragraph_type,
        'icon' => $this->getParagraphTypeIcon($paragraph_type),
        'content_types' => [],
        'paths' => [],
        'entities' => [],
      ];
    }

    // Scan all entity types for paragraph references.
    $this->scanEntityReferences($usage_data);

    // Filter by path search if provided.
    if (!empty($path_search)) {
      $usage_data = $this->filterByPath($usage_data, $path_search);
    }

    return $usage_data;
  }

  /**
   * Scan all entity types for paragraph field references.
   *
   * @param array &$usage_data
   *   Usage data array to populate.
   */
  protected function scanEntityReferences(array &$usage_data) {
    $entity_types = $this->entityTypeManager->getDefinitions();

    foreach ($entity_types as $entity_type_id => $entity_type) {
      // Skip non-fieldable entities and configuration entities.
      if (!$entity_type->entityClassImplements('\Drupal\Core\Entity\FieldableEntityInterface')) {
        continue;
      }

      // Skip entity types that don't support base fields.
      if (!$entity_type->hasKey('id')) {
        continue;
      }

      try {
        $bundles = [];

        if ($entity_type->getBundleEntityType()) {
          $bundles = $this->entityTypeManager
            ->getStorage($entity_type->getBundleEntityType())
            ->loadMultiple();
        }
        elseif (!$entity_type->hasKey('bundle')) {
          // No bundle key, treat entity type itself as bundle.
          $bundles = [$entity_type_id => $entity_type_id];
        }

        foreach ($bundles as $bundle_id => $bundle) {
          $this->scanBundle($entity_type_id, $bundle_id, $usage_data);
        }
      }
      catch (\Exception $e) {
        // Skip entity types that cause errors.
        continue;
      }
    }
  }

  /**
   * Scan a specific bundle for paragraph references.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param array &$usage_data
   *   Usage data array to populate.
   */
  protected function scanBundle($entity_type_id, $bundle_id, array &$usage_data) {
    try {
      $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);

      foreach ($field_definitions as $field_name => $field_definition) {
        // Check if this is an entity_reference_revisions field targeting paragraphs.
        if ($field_definition->getType() === 'entity_reference_revisions') {
          $settings = $field_definition->getSettings();
          $target_type = $settings['target_type'] ?? NULL;

          if ($target_type === 'paragraph') {
            $this->processParagraphField($entity_type_id, $bundle_id, $field_name, $usage_data);
          }
        }
      }
    }
    catch (\Exception $e) {
      // Skip bundles that cause errors.
    }
  }

  /**
   * Process a paragraph field to extract usage data.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param string $field_name
   *   The field name.
   * @param array &$usage_data
   *   Usage data array to populate.
   */
  protected function processParagraphField($entity_type_id, $bundle_id, $field_name, array &$usage_data) {
    try {
      $storage = $this->entityTypeManager->getStorage($entity_type_id);
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      $bundle_key = $entity_type->getKey('bundle');

      $query = $storage->getQuery()
        ->accessCheck(FALSE);

      if ($bundle_key) {
        $query->condition($bundle_key, $bundle_id);
      }

      $entity_ids = $query->execute();

      if (empty($entity_ids)) {
        return;
      }

      $entities = $storage->loadMultiple($entity_ids);

      foreach ($entities as $entity) {
        if (!$entity->hasField($field_name)) {
          continue;
        }

        $field_items = $entity->get($field_name);

        foreach ($field_items as $field_item) {
          $paragraph = $field_item->entity;
          if (!$paragraph) {
            continue;
          }

          $paragraph_type_id = $paragraph->bundle();

          if (!isset($usage_data[$paragraph_type_id])) {
            continue;
          }

          // Add content type.
          $bundle_label = $this->getBundleLabel($entity_type_id, $bundle_id);
          $usage_data[$paragraph_type_id]['content_types'][$bundle_label] = $bundle_label;

          // Add path alias.
          $path_alias = $this->getEntityPathAlias($entity);
          if ($path_alias) {
            $usage_data[$paragraph_type_id]['paths'][$path_alias] = $path_alias;
          }

          // Store entity reference for detail view.
          $usage_data[$paragraph_type_id]['entities'][] = [
            'entity' => $entity,
            'field_name' => $field_name,
            'paragraph' => $paragraph,
          ];
        }
      }
    }
    catch (\Exception $e) {
      // Silently skip entities that cause errors.
    }
  }

  /**
   * Get the path alias for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return string|null
   *   The path alias or NULL.
   */
  public function getEntityPathAlias($entity) {
    if (!$entity->hasLinkTemplate('canonical')) {
      return NULL;
    }

    try {
      $url = $entity->toUrl('canonical');
      $internal_path = $url->getInternalPath();

      if ($internal_path) {
        $alias = $this->pathAliasManager->getAliasByPath('/' . $internal_path);
        return $alias !== '/' . $internal_path ? $alias : '/' . $internal_path;
      }
    }
    catch (\Exception $e) {
      // Return NULL if unable to get path.
    }

    return NULL;
  }

  /**
   * Get the bundle label.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return string
   *   The bundle label.
   */
  public function getBundleLabel($entity_type_id, $bundle_id) {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    $bundle_entity_type = $entity_type->getBundleEntityType();

    if ($bundle_entity_type) {
      $bundle_entity = $this->entityTypeManager
        ->getStorage($bundle_entity_type)
        ->load($bundle_id);

      if ($bundle_entity) {
        return $bundle_entity->label();
      }
    }

    return $bundle_id;
  }

  /**
   * Get detailed usage for a specific paragraph type.
   *
   * @param string $paragraph_type_id
   *   The paragraph type ID.
   *
   * @return array
   *   Detailed usage information.
   */
  public function getDetailedUsage($paragraph_type_id) {
    $usage_data = $this->collectUsageData([$paragraph_type_id]);
    return $usage_data[$paragraph_type_id] ?? [];
  }

  /**
   * Filter usage data by path search.
   *
   * @param array $usage_data
   *   The usage data to filter.
   * @param string $path_search
   *   The path search string.
   *
   * @return array
   *   Filtered usage data.
   */
  protected function filterByPath(array $usage_data, $path_search) {
    $path_search = strtolower(trim($path_search));
    $filtered_data = [];

    foreach ($usage_data as $type_id => $data) {
      // Check if any path matches the search.
      $matching_paths = [];
      $matching_entities = [];

      if (!empty($data['paths'])) {
        foreach ($data['paths'] as $path) {
          if (stripos($path, $path_search) !== FALSE) {
            $matching_paths[$path] = $path;
          }
        }
      }

      // Filter entities to only include those with matching paths.
      if (!empty($data['entities']) && !empty($matching_paths)) {
        foreach ($data['entities'] as $entity_data) {
          $entity_path = $this->getEntityPathAlias($entity_data['entity']);
          if ($entity_path && isset($matching_paths[$entity_path])) {
            $matching_entities[] = $entity_data;
          }
        }
      }

      // Only include this paragraph type if it has matching paths.
      if (!empty($matching_paths)) {
        $filtered_data[$type_id] = $data;
        $filtered_data[$type_id]['paths'] = $matching_paths;
        $filtered_data[$type_id]['entities'] = $matching_entities;

        // Update content types to only include those from matching entities.
        $content_types = [];
        foreach ($matching_entities as $entity_data) {
          $entity = $entity_data['entity'];
          $bundle_label = $this->getBundleLabel(
            $entity->getEntityTypeId(),
            $entity->bundle()
          );
          $content_types[$bundle_label] = $bundle_label;
        }
        $filtered_data[$type_id]['content_types'] = $content_types;
      }
    }

    return $filtered_data;
  }

}
