<?php

namespace Drupal\paragraph_usage_dashboard\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\State\StateInterface;
use Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Pager\PagerParametersInterface;


/**
 * Controller for the paragraph usage dashboard.
 */
class DashboardController extends ControllerBase {

  /**
   * The paragraph usage collector service.
   *
   * @var \Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector
   */
  protected $usageCollector;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

    /**
   * Pager manager.
   *
   * @var \Drupal\Core\Pager\PagerManagerInterface
   */
  protected PagerManagerInterface $pagerManager;

  /**
   * Pager parameters.
   *
   * @var \Drupal\Core\Pager\PagerParametersInterface
   */
  protected PagerParametersInterface $pagerParameters;


  /**
   * Constructs a DashboardController object.
   *
   * @param \Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector $usage_collector
   *   The paragraph usage collector service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    ParagraphUsageCollector $usage_collector,
    StateInterface $state,
    PagerManagerInterface $pagerManager,
    PagerParametersInterface $pagerParameters,
  ) {
    $this->usageCollector = $usage_collector;
    $this->state = $state;
    $this->pagerManager = $pagerManager;
    $this->pagerParameters = $pagerParameters;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('paragraph_usage_dashboard.collector'),
      $container->get('state'),
      $container->get('pager.manager'),
      $container->get('pager.parameters'),
    );
  }

  /**
   * Displays the paragraph usage dashboard.
   *
   * @return array
   *   A render array.
   */
  public function dashboard() {
    $selected_types = $this->state->get('paragraph_usage_dashboard.selected_types', []);
    $path_search = $this->state->get('paragraph_usage_dashboard.path_search', '');
    $name_search = $this->state->get('paragraph_usage_dashboard.name_search', '');
    $show_only_unused = $this->state->get('paragraph_usage_dashboard.show_only_unused', FALSE);
    
    $usage_data = $this->usageCollector->collectUsageData($selected_types, $path_search, $name_search);
    
    if ($show_only_unused) {
      $usage_data = array_filter($usage_data, function($data) {
        return empty($data['entities']);
      });
    }
    
    $items_per_page = 10;
    $total = count($usage_data);

    $current_page = $this->pagerParameters->findPage();
    $from = $current_page * $items_per_page;

    $pager = $this->pagerManager->createPager($total, $items_per_page);
    $usage_data = array_slice($usage_data, $from, $items_per_page, TRUE);


    // Build filter form.
    $filter_form = $this->formBuilder()->getForm('Drupal\paragraph_usage_dashboard\Form\ParagraphFilterForm');


    // Build table rows.
    $rows = [];
    foreach ($usage_data as $type_id => $data) {
      $icon_render = [];
      if ($data['icon']) {
        $file_url = \Drupal::service('file_url_generator')->generateAbsoluteString($data['icon']->getFileUri());
        $icon_render = [
          '#type' => 'link',
          '#title' => [
            '#theme' => 'image_style',
            '#style_name' => 'thumbnail',
            '#uri' => $data['icon']->getFileUri(),
            '#alt' => $this->t('Icon for @type', ['@type' => $data['type']->label()]),
            '#width' => 50,
            '#height' => 50,
          ],
          '#url' => \Drupal\Core\Url::fromRoute('paragraph_usage_dashboard.image_modal', [
            'file_url' => base64_encode($file_url),
          ]),
          '#attributes' => [
            'class' => ['use-ajax'],
            'data-dialog-type' => 'modal',
            'data-dialog-options' => json_encode(['width' => 'auto']),
          ],
        ];
      }

      $content_types = !empty($data['content_types']) ? implode(', ', $data['content_types']) : $this->t('None');

      if (!empty($data['nested_in_paragraphs'])) {
        $nested_info = $this->t('Nested in: @types', [
          '@types' => implode(', ', $data['nested_in_paragraphs'])
        ]);
        $content_types .= '<br><em style="font-size: 0.9em; color: #666;">' . $nested_info . '</em>';
      }

      // Build path links.
      $path_links = [];
      if (!empty($data['paths'])) {
        $displayed_paths = array_slice($data['paths'], 0, 5, TRUE);
        $total_paths = count($data['paths']);
        
        foreach ($displayed_paths as $path) {
          // More robust validation for internal Drupal paths.
          // Valid internal paths must:
          // 1. Start with exactly one forward slash
          // 2. Not contain protocol indicators (://)
          // 3. Not start with special characters like colons (excludes SharePoint paths like :p:/)
          $is_valid_internal = (
            strpos($path, '/') === 0 &&
            strpos($path, '://') === FALSE &&
            !preg_match('/^:[a-z]:/', $path) &&
            strlen($path) > 1
          );

          if ($is_valid_internal) {
            // Wrap in try-catch to handle any unexpected URL generation errors.
            try {
              $path_links[] = [
                '#type' => 'link',
                '#title' => $path,
                '#url' => \Drupal\Core\Url::fromUserInput($path),
              ];
            }
            catch (\Exception $e) {
              // If URL generation fails, display as plain text.
              $path_links[] = ['#markup' => $path];
            }
          }
          else {
            // Display as plain text for external, SharePoint, or invalid paths.
            $path_links[] = ['#markup' => $path];
          }
          $path_links[] = ['#markup' => ', '];
        }
        // Remove last comma.
        array_pop($path_links);
        
        // Add "and X more" if there are more paths.
        if ($total_paths > 5) {
          $remaining = $total_paths - 5;
          $path_links[] = ['#markup' => ' ' . $this->t('and @count more', ['@count' => $remaining])];
        }
      }
      else {
        $path_links = ['#markup' => $this->t('None')];
      }

      // Check if paragraph type is unused.
      $is_unused = empty($data['entities']);

      // Link paragraph type name to backend configuration.
      $paragraph_type_name = [
        '#type' => 'container',
      ];

      $paragraph_type_name['link'] = [
        '#type' => 'link',
        '#title' => $data['type']->label(),
        '#url' => \Drupal\Core\Url::fromRoute('entity.paragraphs_type.edit_form', [
          'paragraphs_type' => $type_id,
        ]),
      ];

      // Add unused badge if applicable.
      if ($is_unused) {
        $paragraph_type_name['badge'] = [
          '#markup' => ' <span class="paragraph-unused-badge">' . $this->t('Unused') . '</span>',
        ];
      }

      // Add "View Usage Details" button.
      $detail_button = [
        '#type' => 'link',
        '#title' => $this->t('View Usage Details'),
        '#url' => \Drupal\Core\Url::fromRoute('paragraph_usage_dashboard.detail', [
          'paragraph_type' => $type_id,
        ]),
        '#attributes' => ['class' => ['button', 'button--small', 'button--primary']],
      ];

      $rows[] = [
        ['data' => $icon_render],
        ['data' => $paragraph_type_name],
        ['data' => ['#markup' => $content_types]],
        ['data' => $path_links],
        ['data' => $detail_button],
      ];
    }

    $build = [];

    $build['filter'] = $filter_form;
    
    $build['cache_clear'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['paragraph-usage-cache-clear']],
      'link' => [
        '#type' => 'link',
        '#title' => $this->t('Clear Cache'),
        '#url' => \Drupal\Core\Url::fromRoute('paragraph_usage_dashboard.clear_cache'),
        '#attributes' => [
          'class' => ['button', 'button--small'],
        ],
      ],
      'description' => [
        '#markup' => '<p class="description">' . $this->t('Clear the usage data cache to refresh the dashboard with latest data.') . '</p>',
      ],
    ];
    
    $build['table'] = [
      '#theme' => 'table',
      '#header' => [
        $this->t('Icon'),
        $this->t('Paragraph Type'),
        $this->t('Used in Content Type(s)'),
        $this->t('Path Alias(es)'),
        $this->t('Usage Details'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No paragraph usage found.'),
      '#attributes' => ['class' => ['paragraph-usage-dashboard-table']],
      '#cache' => [
        'contexts' => ['url.query_args:page'],
        'tags' => ['paragraph_list', 'node_list', 'paragraph_usage_dashboard'],
        'max-age' => 3600, // 1 hour
      ],
    ];

    $build['#attached']['library'][] = 'paragraph_usage_dashboard/dashboard';
    $build['#attached']['library'][] = 'core/drupal.dialog.ajax';

    $build['pager'] = [
      '#type' => 'pager',
      '#quantity' => 5,
    ];

    return $build;
  }

  /**
   * Displays an image in a modal dialog.
   *
   * @param string $file_url
   *   The base64 encoded file URL.
   *
   * @return array
   *   A render array.
   */
  public function imageModal($file_url) {
    $decoded_url = base64_decode($file_url);

    return [
      '#theme' => 'image',
      '#uri' => $decoded_url,
      '#alt' => $this->t('Paragraph icon preview'),
      '#attributes' => [
        'style' => 'max-width: 100%; height: auto;',
      ],
    ];
  }

  /**
   * Clear the usage data cache.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect back to dashboard.
   */
  public function clearCache() {
    $this->usageCollector->clearCache();
    $this->messenger()->addStatus($this->t('Usage data cache has been cleared.'));
    return $this->redirect('paragraph_usage_dashboard.dashboard');
  }

}
