<?php

namespace Drupal\paragraph_usage_dashboard\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\file\FileUsage\FileUsageInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Database\Connection;

/**
 * Service to collect paragraph usage data across the site.
 */
class ParagraphUsageCollector {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The path alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $pathAliasManager;

  /**
   * The file usage service.
   *
   * @var \Drupal\file\FileUsage\FileUsageInterface
   */
  protected $fileUsage;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a ParagraphUsageCollector object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\path_alias\AliasManagerInterface $path_alias_manager
   *   The path alias manager.
   * @param \Drupal\file\FileUsage\FileUsageInterface $file_usage
   *   The file usage service.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    AliasManagerInterface $path_alias_manager,
    FileUsageInterface $file_usage,
    EntityRepositoryInterface $entity_repository,
    CacheBackendInterface $cache,
    Connection $database
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->pathAliasManager = $path_alias_manager;
    $this->fileUsage = $file_usage;
    $this->entityRepository = $entity_repository;
    $this->cache = $cache;
    $this->database = $database;
  }

  /**
   * Get all paragraph types.
   *
   * @return array
   *   Array of paragraph type entities keyed by ID.
   */
  public function getAllParagraphTypes() {
    return $this->entityTypeManager
      ->getStorage('paragraphs_type')
      ->loadMultiple();
  }

  /**
   * Get the icon file entity for a paragraph type.
   *
   * @param \Drupal\paragraphs\Entity\ParagraphsType $paragraph_type
   *   The paragraph type entity.
   *
   * @return \Drupal\file\FileInterface|null
   *   The file entity or NULL if not found.
   */
  public function getParagraphTypeIcon($paragraph_type) {
    $icon_uuid = $paragraph_type->get('icon_uuid');
    if (empty($icon_uuid)) {
      return NULL;
    }

    $files = $this->entityTypeManager
      ->getStorage('file')
      ->loadByProperties(['uuid' => $icon_uuid]);

    return $files ? reset($files) : NULL;
  }

  /**
   * Collect usage data for all paragraph types.
   *
   * @param array $selected_types
   *   Array of selected paragraph type IDs to filter by.
   * @param string $path_search
   *   Optional path search string to filter by path aliases.
   * @param string $name_search
   *   Optional name search string to filter by paragraph type name.
   *
   * @return array
   *   Array of usage data keyed by paragraph type ID.
   */
  public function collectUsageData(array $selected_types = [], $path_search = '', $name_search = '') {
    // Create cache key based on filters.
    $cache_key = 'paragraph_usage_dashboard:usage_data:' . md5(serialize([
      'selected_types' => $selected_types,
      'path_search' => $path_search,
      'name_search' => $name_search,
    ]));

    // Try to get from cache first.
    $cached = $this->cache->get($cache_key);
    if ($cached) {
      return $cached->data;
    }

    $usage_data = [];
    $paragraph_types = $this->getAllParagraphTypes();

    foreach ($paragraph_types as $type_id => $paragraph_type) {
      // Filter by selected types if provided.
      if (!empty($selected_types) && !in_array($type_id, $selected_types)) {
        continue;
      }

      // Filter by name search if provided.
      if (!empty($name_search)) {
        $label = strtolower($paragraph_type->label());
        $search = strtolower(trim($name_search));
        if (stripos($label, $search) === FALSE) {
          continue;
        }
      }

      $usage_data[$type_id] = [
        'type' => $paragraph_type,
        'icon' => $this->getParagraphTypeIcon($paragraph_type),
        'content_types' => [],
        'paths' => [],
        'entities' => [],
        'nested_in_paragraphs' => [], // Track which paragraph types contain this one
      ];
    }

    // Scan all entity types for paragraph references.
    $this->scanEntityReferencesOptimized($usage_data);

    $this->scanNestedParagraphReferences($usage_data);

    // Filter by path search if provided.
    if (!empty($path_search)) {
      $usage_data = $this->filterByPath($usage_data, $path_search);
    }

    // Cache for 1 hour with proper cache tags.
    $cache_tags = ['paragraph_list', 'node_list', 'paragraph_usage_dashboard'];
    $this->cache->set($cache_key, $usage_data, time() + 3600, $cache_tags);

    return $usage_data;
  }

  /**
   * Optimized scan using database queries instead of loading entities.
   *
   * @param array &$usage_data
   *   Usage data array to populate.
   */
  protected function scanEntityReferencesOptimized(array &$usage_data) {
    // Build a map of paragraph type IDs to track.
    $paragraph_type_ids = array_keys($usage_data);
    
    if (empty($paragraph_type_ids)) {
      return;
    }

    $entity_types = $this->entityTypeManager->getDefinitions();

    foreach ($entity_types as $entity_type_id => $entity_type) {
      // Skip non-fieldable entities and configuration entities.
      if (!$entity_type->entityClassImplements('\Drupal\Core\Entity\FieldableEntityInterface')) {
        continue;
      }

      // Skip entity types that don't support base fields.
      if (!$entity_type->hasKey('id')) {
        continue;
      }

      try {
        $bundles = [];

        if ($entity_type->getBundleEntityType()) {
          $bundles = $this->entityTypeManager
            ->getStorage($entity_type->getBundleEntityType())
            ->loadMultiple();
        }
        elseif (!$entity_type->hasKey('bundle')) {
          // No bundle key, treat entity type itself as bundle.
          $bundles = [$entity_type_id => $entity_type_id];
        }

        foreach ($bundles as $bundle_id => $bundle) {
          $this->scanBundleOptimized($entity_type_id, $bundle_id, $usage_data, $paragraph_type_ids);
        }
      }
      catch (\Exception $e) {
        // Skip entity types that cause errors.
        continue;
      }
    }
  }

  /**
   * Optimized scan of a specific bundle using database queries.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param array &$usage_data
   *   Usage data array to populate.
   * @param array $paragraph_type_ids
   *   Array of paragraph type IDs to track.
   */
  protected function scanBundleOptimized($entity_type_id, $bundle_id, array &$usage_data, array $paragraph_type_ids) {
    try {
      $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);

      foreach ($field_definitions as $field_name => $field_definition) {
        // Check if this is an entity_reference_revisions field targeting paragraphs.
        if ($field_definition->getType() === 'entity_reference_revisions') {
          $settings = $field_definition->getSettings();
          $target_type = $settings['target_type'] ?? NULL;

          if ($target_type === 'paragraph') {
            $this->processParagraphFieldOptimized($entity_type_id, $bundle_id, $field_name, $usage_data, $paragraph_type_ids);
          }
        }
      }
    }
    catch (\Exception $e) {
      // Skip bundles that cause errors.
    }
  }

  /**
   * Process a paragraph field using optimized queries.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param string $field_name
   *   The field name.
   * @param array &$usage_data
   *   Usage data array to populate.
   * @param array $paragraph_type_ids
   *   Array of paragraph type IDs to track.
   */
  protected function processParagraphFieldOptimized($entity_type_id, $bundle_id, $field_name, array &$usage_data, array $paragraph_type_ids) {
    try {
      $storage = $this->entityTypeManager->getStorage($entity_type_id);
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      $bundle_key = $entity_type->getKey('bundle');

      // Build query to find entities with this field.
      $query = $storage->getQuery()
        ->accessCheck(FALSE)
        ->exists($field_name);

      if ($bundle_key) {
        $query->condition($bundle_key, $bundle_id);
      }

      // Limit initial query to reduce memory usage.
      $entity_ids = $query->execute();

      if (empty($entity_ids)) {
        return;
      }

      // Process in batches of 50 to avoid memory issues.
      $batch_size = 50;
      $entity_id_chunks = array_chunk($entity_ids, $batch_size, TRUE);

      foreach ($entity_id_chunks as $chunk) {
        $entities = $storage->loadMultiple($chunk);

        foreach ($entities as $entity) {
          if (!$entity->hasField($field_name)) {
            continue;
          }

          $field_items = $entity->get($field_name);

          foreach ($field_items as $field_item) {
            $paragraph = $field_item->entity;
            if (!$paragraph) {
              continue;
            }

            $paragraph_type_id = $paragraph->bundle();

            // Only track paragraph types we care about.
            if (!in_array($paragraph_type_id, $paragraph_type_ids)) {
              continue;
            }

            if (!isset($usage_data[$paragraph_type_id])) {
              continue;
            }

            // Add content type.
            $bundle_label = $this->getBundleLabel($entity_type_id, $bundle_id);
            $usage_data[$paragraph_type_id]['content_types'][$bundle_label] = $bundle_label;

            // Add path alias (cached).
            $path_alias = $this->getEntityPathAliasCached($entity);
            if ($path_alias) {
              $usage_data[$paragraph_type_id]['paths'][$path_alias] = $path_alias;
            }

            // Store minimal entity reference (ID only) for detail view.
            $usage_data[$paragraph_type_id]['entities'][] = [
              'entity_type_id' => $entity_type_id,
              'entity_id' => $entity->id(),
              'field_name' => $field_name,
              'paragraph_id' => $paragraph->id(),
            ];
          }
        }

        // Clear entity cache after each batch.
        $storage->resetCache($chunk);
      }
    }
    catch (\Exception $e) {
      // Silently skip entities that cause errors.
    }
  }

  /**
   * Get the path alias for an entity with caching.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return string|null
   *   The path alias or NULL.
   */
  protected function getEntityPathAliasCached($entity) {
    $cache_key = 'paragraph_usage_dashboard:path_alias:' . $entity->getEntityTypeId() . ':' . $entity->id();
    
    $cached = $this->cache->get($cache_key);
    if ($cached) {
      return $cached->data;
    }

    $path_alias = $this->getEntityPathAlias($entity);
    
    // Cache path alias for 24 hours.
    $this->cache->set($cache_key, $path_alias, time() + 86400, [
      $entity->getEntityTypeId() . ':' . $entity->id(),
    ]);

    return $path_alias;
  }

  /**
   * Get the path alias for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return string|null
   *   The path alias or NULL.
   */
  public function getEntityPathAlias($entity) {
    if (!$entity->hasLinkTemplate('canonical')) {
      return NULL;
    }

    try {
      $url = $entity->toUrl('canonical');

      // Skip external URLs.
      if ($url->isExternal()) {
        return NULL;
      }

      $internal_path = $url->getInternalPath();

      // Validate that internal path doesn't contain external URL patterns.
      if ($internal_path && $this->isValidInternalPath($internal_path)) {
        $alias = $this->pathAliasManager->getAliasByPath('/' . $internal_path);
        return $alias !== '/' . $internal_path ? $alias : '/' . $internal_path;
      }
    }
    catch (\Exception $e) {
      // Return NULL if unable to get path.
    }

    return NULL;
  }

  /**
   * Check if a path is a valid internal path.
   *
   * @param string $path
   *   The path to check.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  protected function isValidInternalPath($path) {
    // Reject paths that look like external URLs or invalid patterns.
    $invalid_patterns = [
      '/^:/',           // Starts with colon (SharePoint URLs).
      '/^https?:\/\//', // HTTP/HTTPS URLs.
      '/^\/\//',        // Protocol-relative URLs.
      '/^mailto:/',     // Email links.
      '/^tel:/',        // Phone links.
      '/^ftp:/',        // FTP links.
    ];

    foreach ($invalid_patterns as $pattern) {
      if (preg_match($pattern, $path)) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Get the bundle label.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return string
   *   The bundle label.
   */
  public function getBundleLabel($entity_type_id, $bundle_id) {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    $bundle_entity_type = $entity_type->getBundleEntityType();

    if ($bundle_entity_type) {
      $bundle_entity = $this->entityTypeManager
        ->getStorage($bundle_entity_type)
        ->load($bundle_id);

      if ($bundle_entity) {
        return $bundle_entity->label();
      }
    }

    return $bundle_id;
  }

  /**
   * Get detailed usage for a specific paragraph type.
   *
   * @param string $paragraph_type_id
   *   The paragraph type ID.
   *
   * @return array
   *   Detailed usage information.
   */
  public function getDetailedUsage($paragraph_type_id) {
    $usage_data = $this->collectUsageData([$paragraph_type_id]);
    return $usage_data[$paragraph_type_id] ?? [];
  }

  /**
   * Filter usage data by path search.
   *
   * @param array $usage_data
   *   The usage data to filter.
   * @param string $path_search
   *   The path search string.
   *
   * @return array
   *   Filtered usage data.
   */
  protected function filterByPath(array $usage_data, $path_search) {
    $path_search = strtolower(trim($path_search));
    $filtered_data = [];

    foreach ($usage_data as $type_id => $data) {
      // Check if any path matches the search.
      $matching_paths = [];
      $matching_entities = [];

      if (!empty($data['paths'])) {
        foreach ($data['paths'] as $path) {
          if (stripos($path, $path_search) !== FALSE) {
            $matching_paths[$path] = $path;
          }
        }
      }

      // Filter entities to only include those with matching paths.
      if (!empty($data['entities']) && !empty($matching_paths)) {
        foreach ($data['entities'] as $entity_data) {
          // Load entity to check path.
          $entity = $this->entityTypeManager
            ->getStorage($entity_data['entity_type_id'])
            ->load($entity_data['entity_id']);
          
          if ($entity) {
            $entity_path = $this->getEntityPathAliasCached($entity);
            if ($entity_path && isset($matching_paths[$entity_path])) {
              $matching_entities[] = $entity_data;
            }
          }
        }
      }

      // Only include this paragraph type if it has matching paths.
      if (!empty($matching_paths)) {
        $filtered_data[$type_id] = $data;
        $filtered_data[$type_id]['paths'] = $matching_paths;
        $filtered_data[$type_id]['entities'] = $matching_entities;

        // Update content types to only include those from matching entities.
        $content_types = [];
        foreach ($matching_entities as $entity_data) {
          $entity = $this->entityTypeManager
            ->getStorage($entity_data['entity_type_id'])
            ->load($entity_data['entity_id']);
          
          if ($entity) {
            $bundle_label = $this->getBundleLabel(
              $entity->getEntityTypeId(),
              $entity->bundle()
            );
            $content_types[$bundle_label] = $bundle_label;
          }
        }
        $filtered_data[$type_id]['content_types'] = $content_types;
      }
    }

    return $filtered_data;
  }

  /**
   * Clear all cached usage data.
   */
  public function clearCache() {
    $this->cache->invalidateTags(['paragraph_usage_dashboard']);
  }

  /**
   * Scan for paragraphs that reference other paragraphs (nested paragraphs).
   *
   * @param array &$usage_data
   *   Usage data array to populate.
   */
  protected function scanNestedParagraphReferences(array &$usage_data) {
    $paragraph_type_ids = array_keys($usage_data);
    
    if (empty($paragraph_type_ids)) {
      return;
    }

    try {
      // Get all paragraph types to scan through
      $paragraph_types = $this->getAllParagraphTypes();

      foreach ($paragraph_types as $parent_type_id => $parent_type) {
        // Get field definitions for this paragraph type
        $field_definitions = $this->entityFieldManager->getFieldDefinitions('paragraph', $parent_type_id);

        foreach ($field_definitions as $field_name => $field_definition) {
          // Check if this is an entity_reference_revisions field targeting paragraphs
          if ($field_definition->getType() === 'entity_reference_revisions') {
            $settings = $field_definition->getSettings();
            $target_type = $settings['target_type'] ?? NULL;

            if ($target_type === 'paragraph') {
              // This paragraph type can contain other paragraphs
              // Now find all instances of this parent paragraph type
              $this->processNestedParagraphField($parent_type_id, $field_name, $usage_data, $paragraph_type_ids);
            }
          }
        }
      }
    }
    catch (\Exception $e) {
      // Silently skip if errors occur
    }
  }

  /**
   * Process nested paragraph field to find child paragraph references.
   *
   * @param string $parent_paragraph_type_id
   *   The parent paragraph type ID that contains other paragraphs.
   * @param string $field_name
   *   The field name containing paragraph references.
   * @param array &$usage_data
   *   Usage data array to populate.
   * @param array $paragraph_type_ids
   *   Array of paragraph type IDs to track.
   */
  protected function processNestedParagraphField($parent_paragraph_type_id, $field_name, array &$usage_data, array $paragraph_type_ids) {
    try {
      $paragraph_storage = $this->entityTypeManager->getStorage('paragraph');

      // Query all paragraphs of the parent type
      $query = $paragraph_storage->getQuery()
        ->accessCheck(FALSE)
        ->condition('type', $parent_paragraph_type_id)
        ->exists($field_name);

      $paragraph_ids = $query->execute();

      if (empty($paragraph_ids)) {
        return;
      }

      // Process in batches to avoid memory issues
      $batch_size = 50;
      $paragraph_id_chunks = array_chunk($paragraph_ids, $batch_size, TRUE);

      foreach ($paragraph_id_chunks as $chunk) {
        $parent_paragraphs = $paragraph_storage->loadMultiple($chunk);

        foreach ($parent_paragraphs as $parent_paragraph) {
          if (!$parent_paragraph->hasField($field_name)) {
            continue;
          }

          $field_items = $parent_paragraph->get($field_name);

          foreach ($field_items as $field_item) {
            $child_paragraph = $field_item->entity;
            if (!$child_paragraph) {
              continue;
            }

            $child_type_id = $child_paragraph->bundle();

            // Only track paragraph types we care about
            if (!in_array($child_type_id, $paragraph_type_ids)) {
              continue;
            }

            if (!isset($usage_data[$child_type_id])) {
              continue;
            }

            $parent_label = $this->getBundleLabel('paragraph', $parent_paragraph_type_id);
            $usage_data[$child_type_id]['nested_in_paragraphs'][$parent_label] = $parent_label;

            // This prevents it from being marked as "unused"
            $usage_data[$child_type_id]['entities'][] = [
              'entity_type_id' => 'paragraph',
              'entity_id' => $parent_paragraph->id(),
              'field_name' => $field_name,
              'paragraph_id' => $child_paragraph->id(),
              'is_nested' => TRUE, // Flag to identify nested references
            ];
          }
        }

        // Clear cache after each batch
        $paragraph_storage->resetCache($chunk);
      }
    }
    catch (\Exception $e) {
      // Silently skip entities that cause errors
    }
  }

}
