# Paragraph Usage Dashboard - Complete Module Structure

## Directory Structure

```
paragraph_usage_dashboard/
├── .gitignore
├── CHANGELOG.md
├── DRUPAL_ORG_DESCRIPTION.md
├── INSTALL.txt
├── MODULE_STRUCTURE.md
├── README.md
├── composer.json
├── paragraph_usage_dashboard.info.yml
├── paragraph_usage_dashboard.libraries.yml
├── paragraph_usage_dashboard.links.menu.yml
├── paragraph_usage_dashboard.module
├── paragraph_usage_dashboard.permissions.yml
├── paragraph_usage_dashboard.routing.yml
├── paragraph_usage_dashboard.services.yml
├── config/
│   └── schema/
│       └── paragraph_usage_dashboard.schema.yml
├── css/
│   └── dashboard.css
├── js/
│   └── dashboard.js
├── src/
│   ├── Controller/
│   │   ├── DashboardController.php
│   │   └── ParagraphDetailController.php
│   ├── Form/
│   │   └── ParagraphFilterForm.php
│   ├── Hook/
│   │   └── ThemeHook.php
│   └── Service/
│       └── ParagraphUsageCollector.php
├── templates/
│   ├── paragraph-usage-dashboard.html.twig
│   └── paragraph-usage-detail.html.twig
└── tests/
    └── src/
        └── Functional/
            └── ParagraphUsageDashboardTest.php
```

## File Descriptions

### Root Level Configuration Files

- **.gitignore** - Git ignore patterns for the module
- **CHANGELOG.md** - Version history and changes
- **DRUPAL_ORG_DESCRIPTION.md** - Complete description for drupal.org project page
- **INSTALL.txt** - Installation instructions
- **MODULE_STRUCTURE.md** - This file, documenting the structure
- **README.md** - Main documentation and usage guide
- **composer.json** - Composer package definition

### Module Definition Files (*.yml)

- **paragraph_usage_dashboard.info.yml** - Module metadata and dependencies
- **paragraph_usage_dashboard.libraries.yml** - CSS/JS library definitions
- **paragraph_usage_dashboard.links.menu.yml** - Admin menu integration
- **paragraph_usage_dashboard.module** - Hook implementations and helper functions
- **paragraph_usage_dashboard.permissions.yml** - Permission definitions
- **paragraph_usage_dashboard.routing.yml** - Route definitions
- **paragraph_usage_dashboard.services.yml** - Service container definitions

### Configuration Schema

- **config/schema/paragraph_usage_dashboard.schema.yml** - Configuration schema definition

### Assets

- **css/dashboard.css** - Styles for dashboard and detail views
- **js/dashboard.js** - JavaScript behaviors and enhancements

### PHP Classes (PSR-4 Autoloaded)

#### Controllers
- **src/Controller/DashboardController.php** - Main dashboard page controller
- **src/Controller/ParagraphDetailController.php** - Detail view controller

#### Forms
- **src/Form/ParagraphFilterForm.php** - Filter form for paragraph type selection

#### Hooks
- **src/Hook/ThemeHook.php** - Theme system hooks (Drupal 11 OOP style)

#### Services
- **src/Service/ParagraphUsageCollector.php** - Core service for collecting usage data

### Templates

- **templates/paragraph-usage-dashboard.html.twig** - Dashboard table template
- **templates/paragraph-usage-detail.html.twig** - Detail view template

### Tests

- **tests/src/Functional/ParagraphUsageDashboardTest.php** - Functional tests

## Key Features Implemented

### 1. Service Layer (ParagraphUsageCollector)
- Scans all entity types and bundles
- Identifies entity_reference_revisions fields targeting paragraphs
- Extracts paragraph type information
- Loads paragraph type icons from file entities
- Generates path aliases for parent entities
- Collects content type usage data
- Provides filtered and detailed usage data

### 2. Controllers
- **DashboardController** - Renders main dashboard with table and filter
- **ParagraphDetailController** - Renders detailed instance view

### 3. Forms
- **ParagraphFilterForm** - Multi-select checkbox filter with state persistence

### 4. Routing
- `/admin/reports/paragraph-usage` - Main dashboard
- `/admin/reports/paragraph-usage/{paragraph_type}` - Detail view

### 5. Permissions
- `access paragraph usage dashboard` - Restricted access control

### 6. Theme System
- OOP hook implementation using PHP attributes
- Twig templates for flexible theming
- CSS for responsive table layouts
- JavaScript for progressive enhancement

### 7. Data Collection
- Scans all fieldable entity types
- Supports all bundles
- Handles missing icons gracefully
- Generates human-readable path aliases
- Groups usage by paragraph type
- Tracks field-level usage

## Architecture Highlights

### Dependency Injection
All services and controllers use constructor injection:
```php
public function __construct(
  ParagraphUsageCollector $usage_collector,
  StateInterface $state
) {
  $this->usageCollector = $usage_collector;
  $this->state = $state;
}
```

### Modern OOP Hooks
Uses Drupal 11's attribute-based hook system:
```php
#[Hook('theme')]
public function theme(): array {
  return [
    'paragraph_usage_dashboard' => [
      'variables' => ['usage_data' => []],
    ],
  ];
}
```

### Service Registration
Services defined in YAML with explicit dependencies:
```yaml
services:
  paragraph_usage_dashboard.collector:
    class: Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector
    arguments:
      - '@entity_type.manager'
      - '@entity_field.manager'
      - '@path_alias.manager'
      - '@file.usage'
      - '@entity.repository'
```

### State Management
Uses State API for filter persistence:
```php
$selected_types = $this->state->get('paragraph_usage_dashboard.selected_types', []);
```

## Testing Strategy

Functional tests cover:
- Permission-based access control
- Menu link presence
- Service registration
- Filter form functionality
- Paragraph type display
- Basic rendering

## Deployment

### For Development
1. Place in `modules/custom/paragraph_usage_dashboard`
2. Enable: `drush en paragraph_usage_dashboard`
3. Grant permissions
4. Access at `/admin/reports/paragraph-usage`

### For Drupal.org Contribution
1. Create project on drupal.org
2. Push to git.drupalcode.org
3. Tag release version
4. Packagist will auto-detect via composer.json
5. Users install via: `composer require drupal/paragraph_usage_dashboard`

## Maintenance Notes

- All code follows Drupal coding standards
- PSR-4 autoloading for classes
- Comprehensive inline documentation
- Semantic versioning in composer.json
- No database schema changes required
- No configuration entities needed
- Stateless operation (uses State API for UI preferences only)

## Extension Points

The module is designed for extensibility:

1. **Custom Collectors** - Extend ParagraphUsageCollector for custom data
2. **Additional Controllers** - Add export or bulk operation controllers
3. **Enhanced Templates** - Override Twig templates in themes
4. **Additional Filters** - Extend filter form for more options
5. **Event Subscribers** - Add event subscribers for custom workflows
6. **Drush Commands** - Add CLI commands for reporting
7. **REST Endpoints** - Expose data via REST for external tools

## Performance Considerations

- Entity queries use access checks appropriately
- Limited to loaded entities (does not load all site content at once)
- Caching can be added via configuration
- Filter reduces data processing when specific types selected
- No expensive JOIN operations
- Uses existing Drupal APIs efficiently
