<?php

namespace Drupal\Tests\paragraph_usage_dashboard\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests for the Paragraph Usage Dashboard.
 *
 * @group paragraph_usage_dashboard
 */
class ParagraphUsageDashboardTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'field',
    'paragraphs',
    'entity_reference_revisions',
    'path_alias',
    'file',
    'paragraph_usage_dashboard',
  ];

  /**
   * A user with permission to access the dashboard.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a user with permission to access the dashboard.
    $this->adminUser = $this->drupalCreateUser([
      'access paragraph usage dashboard',
      'administer paragraphs types',
      'administer nodes',
      'access content overview',
    ]);
  }

  /**
   * Tests dashboard access and basic functionality.
   */
  public function testDashboardAccess() {
    // Test that anonymous users cannot access the dashboard.
    $this->drupalGet('/admin/reports/paragraph-usage');
    $this->assertSession()->statusCodeEquals(403);

    // Test that authenticated users with permission can access.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/reports/paragraph-usage');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Paragraph Usage Dashboard');
  }

  /**
   * Tests the filter form functionality.
   */
  public function testFilterForm() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/reports/paragraph-usage');

    // Check that the filter form is present.
    $this->assertSession()->fieldExists('selected_types[]');
    $this->assertSession()->buttonExists('Apply Filter');
    $this->assertSession()->buttonExists('Reset Filter');
  }

  /**
   * Tests menu link presence.
   */
  public function testMenuLink() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/reports');

    // Check that the menu link appears in the reports section.
    $this->assertSession()->linkExists('Paragraph Usage');
    $this->clickLink('Paragraph Usage');
    $this->assertSession()->addressEquals('/admin/reports/paragraph-usage');
  }

  /**
   * Tests that the service is properly registered.
   */
  public function testServiceRegistration() {
    $collector = \Drupal::service('paragraph_usage_dashboard.collector');
    $this->assertNotNull($collector);
    $this->assertInstanceOf(
      'Drupal\paragraph_usage_dashboard\Service\ParagraphUsageCollector',
      $collector
    );
  }

  /**
   * Tests paragraph type collection.
   */
  public function testParagraphTypeCollection() {
    $this->drupalLogin($this->adminUser);

    // Create a paragraph type.
    $paragraph_type = $this->container->get('entity_type.manager')
      ->getStorage('paragraphs_type')
      ->create([
        'id' => 'test_paragraph',
        'label' => 'Test Paragraph',
      ]);
    $paragraph_type->save();

    // Visit the dashboard.
    $this->drupalGet('/admin/reports/paragraph-usage');

    // Check that the paragraph type appears.
    $this->assertSession()->pageTextContains('Test Paragraph');
  }

}
