# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Overview

Paragraphs Blokkli is a Drupal module that provides a visual, block-based editor interface for editing Drupal Paragraphs without traditional forms. The module uses an immutable, mutation-based architecture with GraphQL as its primary API.

## Development Commands

### Code Quality

```bash
# Run PHPStan static analysis
vendor/bin/phpstan analyse

# Run PHP CodeSniffer for code standards
vendor/bin/phpcs

# Fix coding standards automatically
vendor/bin/phpcbf
```

### Testing

```bash
# Run all kernel tests
vendor/bin/phpunit -c core modules/contrib/paragraphs_blokkli/tests

# Run specific test class
vendor/bin/phpunit -c core modules/contrib/paragraphs_blokkli/tests/src/Kernel/EditStateTest.php

# Run tests with filter
vendor/bin/phpunit -c core --filter testEditState modules/contrib/paragraphs_blokkli/tests
```

## Architecture Overview

### Core Concepts

**Mutation-Based State Management**: The entire editing system is built around discrete, reversible operations called "mutations" (Add, Edit, Remove, Move, Duplicate, etc.). Mutations don't modify entities directly; they create new state representations that can be staged, undone/redone, and eventually published.

**Edit State Entity**: `ParagraphsBlokkliEditState` is the central entity that stores:
- A history array of all applied mutations
- Current history index for undo/redo
- Ownership and granular permissions (view/review/edit)
- Validation errors and constraint violations
- Optional scheduling for deferred publishing

**Paragraph Proxies**: `ParagraphProxy` wraps paragraph entities to provide an immutable editing interface. Mutations operate on proxies rather than directly on Drupal entities.

**GraphQL-First API**: All editing operations are exposed via GraphQL (in `paragraphs_blokkli_graphql` submodule). The frontend communicates exclusively through GraphQL queries and mutations.

### Plugin Systems

The module provides multiple plugin types:

1. **Mutation Plugins** (`Plugin/ParagraphsBlokkli/Mutation/`)
   - 30+ mutation types for paragraph operations
   - Annotation: `@ParagraphMutation`
   - Base: `ParagraphMutationPluginBase`
   - Context-driven parameters using `@ContextDefinition`

2. **Search Plugins** (`paragraphs_blokkli_search` submodule)
   - Pluggable search providers for media, content, library items
   - Base: `ParagraphsBlokkliSearchPluginBase`

3. **Conversion Plugins** (`paragraphs_blokkli_conversion` submodule)
   - Convert one paragraph type to another
   - Base: `ParagraphConversionPluginBase`

### Key Services

- `paragraphs_blokkli.manager` (`ParagraphsBlokkliManager`) - Core manager for edit states and publishing
- `plugin.manager.paragraph_mutation` (`ParagraphMutationPluginManager`) - Manages mutation plugins
- `paragraphs_blokkli.helper` (`ParagraphsBlokkliHelper`) - Utility methods
- `paragraphs_blokkli.config` (`ParagraphsBlokkliConfig`) - Module configuration

### Data Flow

1. **Frontend → Backend**: GraphQL mutation with `pluginId` and `values`
2. **Mutation Execution**: Plugin manager instantiates mutation with context
3. **Proxy Mutation**: Mutations modify paragraph proxies
4. **Edit State Storage**: Mutation added to history (unless preview mode)
5. **Response**: Returns mutated state with validation errors
6. **Publishing**: All mutations executed sequentially, applied to host entity

### Permissions System

Granular, role-based permissions implemented in recent commits:

- `view paragraphs blokkli edit state` - View edit states
- `review paragraphs blokkli edit state` - Review changes (includes view)
- `edit paragraphs blokkli edit state` - Edit/mutate paragraphs
- `publish paragraphs blokkli edit state with validation errors` - Override validation
- `take ownership of paragraphs blokkli edit state` - Transfer ownership
- `administer paragraphs blokkli edit state` - Full admin access

Access control handled by `ParagraphsBlokkliEditStateAccessControlHandler`.

### Submodules

- **paragraphs_blokkli_graphql** - GraphQL API layer (queries, mutations, data producers)
- **paragraphs_blokkli_scheduler** - Scheduling support for deferred publishing (`publish_on` timestamp)
- **paragraphs_blokkli_search** - Search plugin system for media, content, library items
- **paragraphs_blokkli_conversion** - Paragraph type conversion with field mapping
- **paragraphs_blokkli_comment** - Commenting on edit states without edit permission
- **paragraphs_blokkli_library** - Reusable paragraph library management
- **paragraphs_blokkli_fragments** - Fragment/snippet support

## Testing Practices

### Test Structure

Tests are located in `tests/src/Kernel/` and use Drupal's KernelTestBase.

**Key Testing Utilities:**

- `ParagraphsBlokkliTestTrait` - Provides helper methods:
  - `getEditState(FieldableEntityInterface $entity)` - Get edit state for entity
  - `paragraphsBlokkliPublish(EditStateInterface $editState)` - Publish changes
  - `getParagraphsBlokkliTestHelper(EditStateInterface $editState)` - Get test helper

- `ParagraphsBlokkliTestHelper` - Helper class for testing mutations

**Example Test Pattern:**

```php
use Drupal\Tests\paragraphs_blokkli\Kernel\ParagraphsBlokkliTestTrait;

class MyTest extends KernelTestBase {
  use ParagraphsBlokkliTestTrait;

  protected function testMutation() {
    $entity = $this->createNode();
    $editState = $this->getEditState($entity);

    // Apply mutations via manager
    $helper = $this->getParagraphsBlokkliTestHelper($editState);

    // Publish changes
    $this->paragraphsBlokkliPublish($editState);
  }
}
```

## Configuration

### PHP CodeSniffer

The module uses custom PHPCS rules (phpcs.xml.dist) with several Drupal standards relaxed:
- Line length limits disabled
- Function/class comment requirements relaxed
- Inline comment formatting relaxed

### PHPStan

Configuration in `phpstan.neon`:
- Excludes `tests/*/data/*`
- Parallel processing: 4 processes max
- Ignores `\Drupal calls should be avoided` warnings (common in Drupal)

## GraphQL Schema

GraphQL schemas are defined in `.graphqls` files:

- `modules/paragraphs_blokkli_graphql/graphql/paragraphs_blokkli_edit.base.graphqls` - Core schema
- `modules/paragraphs_blokkli_scheduler/graphql/paragraphs_blokkli_scheduler.*.graphqls` - Scheduling extensions

Data producers (resolvers) are in `modules/paragraphs_blokkli_graphql/src/Plugin/GraphQL/DataProducer/`.

## Entity Mappings

**pb_entity_mapping** config entity maps external entities to paragraph types for drag-and-drop workflows:
- Maps media/nodes → paragraph types
- Defines which paragraph to create when dropping an entity
- Configured at `/admin/structure/paragraphs_blokkli/entity_mapping`

## Common Mutation Patterns

When adding new mutations:

1. Create plugin class in `src/Plugin/ParagraphsBlokkli/Mutation/`
2. Use `@ParagraphMutation` annotation with `id` and `label`
3. Define `@ContextDefinition` annotations for parameters
4. Extend `ParagraphMutationPluginBase`
5. Implement `execute(ParagraphMutationContext $context)` method
6. Return mutated context with changes applied
7. Handle errors via `MutationException` subclasses
8. Add corresponding GraphQL schema in paragraphs_blokkli_graphql

## Recent Features (Latest Commits)

- **Paragraph Scheduling** (189183b) - Edit states can be scheduled for future publishing
- **Granular Permissions** (1ccddfa) - View/review/edit permission levels
- **Comment Without Edit** (096e19a) - Reviewers can comment without edit access
- **Node Form Integration** (53d00b3) - Shows message when scheduled edits exist
