<?php

namespace Drupal\paragraphs_blokkli_graphql\Plugin\GraphQL\DataProducer\Paragraph;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\paragraphs\ParagraphInterface;
use Drupal\paragraphs_blokkli\ParagraphsBlokkliConfig;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Determine if a paragraph 'is visible' in non-edit mode.
 *
 * @DataProducer(
 *   id = "paragraphs_blokkli_paragraph_is_visible",
 *   name = @Translation("Paragraph is Visible"),
 *   description = @Translation("Returns TRUE if the paragraph is visible."),
 *   consumes = {
 *     "paragraph" = @ContextDefinition("entity:paragraph",
 *       label = @Translation("The paragraph.")
 *     ),
 *   },
 *   produces = @ContextDefinition("boolean",
 *     label = @Translation("TRUE if the paragraph is visible.")
 *   ),
 * )
 */
class IsVisible extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('paragraphs_blokkli.config'),
    );
  }

  /**
   * IsVisible constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\paragraphs_blokkli\ParagraphsBlokkliConfig $config
   *   The paragraphs blokkli config.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    protected ParagraphsBlokkliConfig $config,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Resolver.
   *
   * @param \Drupal\paragraphs\Entity\Paragraph $paragraph
   *   The paragraph.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $fieldContext
   *   The field context.
   *
   * @return bool
   *   TRUE if the paragraph is visible.
   */
  public function resolve(ParagraphInterface $paragraph, FieldContext $fieldContext): bool {
    // Add cache dependency on the config.
    $fieldContext->addCacheableDependency($this->config);

    // If the hide_unpublished_paragraphs setting is enabled, always hide
    // unpublished paragraphs.
    if ($this->config->hideUnpublishedParagraphs()) {
      return $paragraph->isPublished();
    }

    // Default behavior: show all paragraphs (published and unpublished).
    // This does not override existing visibility behaviour done via access
    // checks. The paragraph entity can never be accessed anyway if the user
    // does not have the 'view' permission.
    return TRUE;
  }

}
