<?php

namespace Drupal\paragraphs_blokkli_scheduler\Plugin\GraphQL\DataProducer;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\scheduler\SchedulerManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Whether the paragraph bundle supports the given scheduler process.
 *
 * @DataProducer(
 *   id = "paragraphs_blokkli_scheduler_supports_process",
 *   name = @Translation("Paragraphs Blokkli Scheduler Supports Process"),
 *   description = @Translation("Determine if a paragraph bundle has the given scheduler field."),
 *   produces = @ContextDefinition("boolean",
 *     label = @Translation("If the field exists.")
 *   ),
 *   consumes = {
 *     "type" = @ContextDefinition("any",
 *       label = @Translation("The entity bundle.")
 *     ),
 *     "process" = @ContextDefinition("string",
 *       label = @Translation("The scheduling process."),
 *     ),
 *   }
 * )
 */
class BundleSupportsScheduleProcess extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('scheduler.manager')
    );
  }

  /**
   * HasSchedulerField constructor.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $plugin_id
   *   The plugin id.
   * @param array $plugin_definition
   *   The plugin definition array.
   * @param \Drupal\scheduler\SchedulerManager $schedulerManager
   *   The scheduler manager.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    protected SchedulerManager $schedulerManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Resolver.
   *
   * @param \Drupal\paragraphs\Entity\ParagraphsType
   *   The paragraph type entity.
   * @param string $process
   *   The scheduling process ('publish' or 'unpublish').
   *
   * @return bool
   *   Whether the given scheduling process is enabled.
   */
  public function resolve(ParagraphsType $type, string $process): bool {
    $bundles = $this->schedulerManager->getEnabledTypes('paragraph', $process);
    return in_array($type->id(), $bundles);
  }

}
