<?php

namespace Drupal\paragraphs_blokkli_transform\Plugin\ParagraphsBlokkli\Mutation;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\paragraphs_blokkli\Exception\MutationBadDataException;
use Drupal\paragraphs_blokkli\ParagraphMutationContextInterface;
use Drupal\paragraphs_blokkli\ParagraphMutationPluginBase;
use Drupal\paragraphs_blokkli\ParagraphsBlokkliHelper;
use Drupal\paragraphs_blokkli_transform\ParagraphHostTransformPluginManager;
use Drupal\paragraphs_blokkli_transform\ParagraphTransformPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Transform the host entity.
 *
 * @ParagraphMutation(
 *   id = "host_transform",
 *   label = @Translation("Apply the given transform plugin to the host."),
 *   description = @Translation("Transforms one or more paragraphs."),
 *   arguments = {
 *     "pluginId" = @ContextDefinition("string",
 *       label = @Translation("The ID of the transform plugin.")
 *     ),
 *     "userConfig" = @ContextDefinition("map",
 *       label = @Translation("The user provided configuration."),
 *       required = FALSE,
 *     ),
 *   }
 * )
 */
class HostTransform extends ParagraphMutationPluginBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('uuid'),
      $container->get('paragraphs_blokkli.helper'),
      $container->get('plugin.manager.paragraph_host_transform')
    );
  }

  /**
   * Transform constructor.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $pluginId
   *   The plugin id.
   * @param array $pluginDefinition
   *   The plugin definition array.
   * @param EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param UuidInterface $uuidHelper
   *   The UUID helper.
   * @param ParagraphsBlokkliHelper $paragraphsBlokkliHelper
   *   The paragraphs blokkli helper.
   * @param ParagraphTransformPluginManager $pluginManager
   *   The plugin manager.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    array $pluginDefinition,
    EntityTypeManagerInterface $entityTypeManager,
    UuidInterface $uuidHelper,
    ParagraphsBlokkliHelper $paragraphsBlokkliHelper,
    protected ParagraphHostTransformPluginManager $pluginManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition, $entityTypeManager, $uuidHelper, $paragraphsBlokkliHelper);
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * Executes the transform plugin on the host entity.
   *
   * @param \Drupal\paragraphs_blokkli\ParagraphMutationContextInterface $context
   *   The paragraph mutation context.
   * @param string $pluginId
   *   The ID of the transform plugin to use.
   * @param array|null $userConfigInput
   *   The provided user configuration.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   *   If the transform plugin cannot be created.
   */
  public function execute(ParagraphMutationContextInterface $context, string $pluginId, array|null $userConfigInput): void {
    // If the transform plugin has been executed already, use this configuration.
    $executedConfiguration = $this->configuration['transform'] ?? [];

    if (!$this->pluginManager->hasDefinition($pluginId)) {
      throw new MutationBadDataException("The transform plugin '$pluginId' does not exist.");
    }

    /** @var \Drupal\paragraphs_blokkli_transform\ParagraphHostTransformPluginBase $instance */
    $instance = $this->pluginManager->createInstance($pluginId, $executedConfiguration);

    $userConfig = [];

    if (!empty($userConfigInput)) {
      foreach ($userConfigInput as $item) {
        $name = $item['name'];
        $value = $item['value'];
        $userConfig[$name] = $value;
      }
    }

    $instance->transform($context, $userConfig);

    $this->configuration['transform'] = $instance->configuration;
  }

}
