<?php

namespace Drupal\paragraphs_blokkli_graphql\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\paragraphs_blokkli\ParagraphsBlokkliConfig;
use Drupal\paragraphs_blokkli\Plugin\Field\FieldType\ParagraphsBlokkliHostOptionsFieldItemList;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Build the props for the blokkli provider component.
 *
 * @DataProducer(
 *   id = "paragraphs_blokkli_props",
 *   name = @Translation("Paragraphs Blokkli Props"),
 *   description = @Translation("Returns the blokkli provider props."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The props.")
 *   ),
 *   consumes = {
 *     "entity" = @ContextDefinition("entity",
 *       label = @Translation("The entity.")
 *     ),
 *   }
 * )
 */
class BlokkliProps extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
      $container->get('paragraphs_blokkli.config'),
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * CurrentUser constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\paragraphs_blokkli\ParagraphsBlokkliConfig $blokkliConfig
   *   The blokkli config.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfo $bundleInfo
   *   The bundle info service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    protected AccountInterface $currentUser,
    protected ParagraphsBlokkliConfig $blokkliConfig,
    protected EntityTypeBundleInfoInterface $bundleInfo,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Resolver.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $fieldContext
   *   The field context.
   *
   * @return array
   *   The blokkli props.
   */
  public function resolve(EntityInterface $entity, FieldContext $fieldContext): array {
    $permissions = $this->getPermissions($entity, $fieldContext);

    $hostOptions = $this->getHostOptions($entity);

    return [
      'permissions' => $permissions,
      'entityType' => $entity->getEntityTypeId(),
      'entityBundle' => $entity->bundle(),
      'entityUuid' => $entity->uuid(),
      'language' => $entity->language()->getId(),
      'editLabel' => $this->getEditLabel($entity, $permissions),
      // Make sure we resolve an empty object if no options exist.
      'hostOptions' => empty($hostOptions) ? new \stdClass() : $hostOptions,
    ];
  }

  /**
   * Determine if the current user can use blokkli on the entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $fieldContext
   *   The field context.
   *
   * @return string[]
   *   The permissions.
   */
  protected function getPermissions(EntityInterface $entity, FieldContext $fieldContext): array {
    $fieldContext->addCacheContexts(['user.permissions']);

    if (!$this->currentUser->hasPermission('view paragraphs blokkli edit state')) {
      return [];
    }

    // Now add the config as a cache dependency.
    $fieldContext->addCacheableDependency($this->blokkliConfig);

    // Entity is not enabled for use with blökkli.
    if (!$this->blokkliConfig->isEnabledForEntity($entity)) {
      return [];
    }

    $permissions = ['view'];

    if ($this->currentUser->hasPermission('review paragraphs blokkli edit state')) {
      $permissions[] = 'review';
    }

    if ($this->currentUser->hasPermission('edit paragraphs blokkli edit state')) {
      $accessResult = $entity->access('update', $this->currentUser, TRUE);
      $fieldContext->addCacheableDependency($accessResult);
      if ($accessResult->isAllowed()) {
        $permissions[] = 'edit';
      }
    }

    return $permissions;
  }

  /**
   * Get the edit label.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param string[] $permissions
   *   The permissions.
   *
   * @return string|null
   *   The edit label.
   */
  protected function getEditLabel(EntityInterface $entity, array $permissions): string|null {
    if (empty($permissions)) {
      return NULL;
    }

    $entityType = $entity->getEntityTypeId();

    $info = $this->bundleInfo->getBundleInfo($entityType);
    $bundleLabel = $info[$entity->bundle()]['label'] ?? NULL;

    if (!$bundleLabel) {
      return NULL;
    }

    if (in_array('edit', $permissions)) {
      return $this->t('Edit @entity', ['@entity' => $bundleLabel]);
    }
    elseif (in_array('review', $permissions)) {
      return $this->t('Review @entity', ['@entity' => $bundleLabel]);
    }

    return $this->t('View @entity', ['@entity' => $bundleLabel]);
  }

  /**
   * Get the options value.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return array|null
   *   The options.
   */
  private function getHostOptions(EntityInterface $entity): array|null {
    if (!$entity instanceof FieldableEntityInterface) {
      return NULL;
    }

    $fieldDefinitions = $entity->getFieldDefinitions();

    foreach ($fieldDefinitions as $definition) {
      if ($definition->getType() === 'paragraphs_blokkli_host_options') {
        $fieldName = $definition->getName();
        $field = $entity->get($fieldName);
        assert($field instanceof ParagraphsBlokkliHostOptionsFieldItemList);
        return $field->getKeyedOptions();
      }
    }

    return NULL;
  }

}
