<?php

namespace Drupal\paragraphs_blokkli_graphql\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\paragraphs_blokkli\ParagraphsBlokkliManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Loads the edit state for mutations.
 *
 * @DataProducer(
 *   id = "paragraphs_blokkli_load_edit_state_for_mutation",
 *   name = @Translation("Paragraphs Blokkli Load Edit State for Mutations"),
 *   description = @Translation("Resolves the edit state for use in mutations."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The edit state.")
 *   ),
 *   consumes = {
 *     "entity" = @ContextDefinition("entity",
 *       label = @Translation("The host entity.")
 *     ),
 *     "preview" = @ContextDefinition("boolean",
 *       label = @Translation("Whether the edit state should be loaded in preview mode."),
 *       required = FALSE,
 *     ),
 *   }
 * )
 */
class LoadEditStateForMutation extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('paragraphs_blokkli.manager'),
    );
  }

  /**
   * LoadEditStateForMutation constructor.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $pluginId
   *   The plugin id.
   * @param array $pluginDefinition
   *   The plugin definition array.
   * @param \Drupal\paragraphs_blokkli\ParagraphsBlokkliManager $blokkliManager
   *   The blokkli manager.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    array $pluginDefinition,
    protected ParagraphsBlokkliManager $blokkliManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * Resolver.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity.
   * @param bool|null $preview
   *   Whether the edit state should be loaded in preview mode.
   */
  public function resolve(FieldableEntityInterface $entity, ?bool $preview = FALSE) {
    $state = $this->blokkliManager->getParagraphsEditState($entity);

    if (!$state) {
      return NULL;
    }

    if ($preview) {
      $state->setMutationPreviewMode();
    }

    return $state;
  }

}
