<?php

namespace Drupal\paragraphs_blokkli_graphql\Plugin\GraphQL\DataProducer\Search;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\paragraphs_blokkli\Entity\ParagraphsBlokkliEditState;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Searches for edit states.
 *
 * @DataProducer(
 *   id = "paragraphs_blokkli_search_edit_states",
 *   name = @Translation("Paragraphs Blokkli Search Edit States"),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The search results.")
 *   ),
 *   consumes = {
 *     "provider" = @ContextDefinition("any",
 *       label = @Translation("The search provider."),
 *     ),
 *     "page" = @ContextDefinition("integer",
 *       label = @Translation("The page."),
 *       required = FALSE,
 *     ),
 *   }
 * )
 */
class EditStates extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('current_user'),
    );
  }

  /**
   * ParagraphsBlokkliMediaLibraryResults constructor.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $pluginId
   *   The plugin id.
   * @param array $pluginDefinition
   *   The plugin definition array.
   * @param \Drupal\Core\Entity\EntityTypeManager $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    array $pluginDefinition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected AccountInterface $currentUser,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * Resolver.
   *
   * @param \Drupal\paragraphs_blokkli_search\ParagraphsBlokkliSearchPluginBase $provider
   * @param int|null $page
   * @param FieldContext|null $fieldContext
   */
  public function resolve($provider, int|null $page, FieldContext|null $fieldContext) {
    /** @var \Drupal\paragraphs_blokkli_search\ParagraphsBlokkliSearchEditStatesInterface $provider */
    $result = $provider->getEditStatesSearchResults($page, $fieldContext);

    return [
      'items' => $this->mapItems($result->ids),
      'perPage' => $result->perPage,
      'total' => $result->total,
    ];
  }

  /**
   * Map result items.
   *
   * @param string[] $ids
   *   The edit state IDs.
   */
  private function mapItems(array $ids): array {
    $storage = $this->entityTypeManager->getStorage('paragraphs_blokkli_edit_state');
    $items = [];

    foreach ($ids as $id) {
      $state = $storage->load($id);
      if (!$state instanceof ParagraphsBlokkliEditState) {
        continue;
      }

      $entity = $state->getHostEntity();
      if (!$entity) {
        continue;
      }

      $items[] = $state;
    }

    return $items;
  }

}
