<?php

namespace Drupal\paragraphs_blokkli_scheduler;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\paragraphs_blokkli\ParagraphsBlokkliManager;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\paragraphs_blokkli\EditStateInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for processing scheduled edit state publishing.
 */
class EditStateSchedulerService {

  /**
   * The logger channel.
   */
  protected LoggerInterface $logger;

  /**
   * Constructs an EditStateSchedulerService object.
   *
   * @param \Drupal\paragraphs_blokkli\ParagraphsBlokkliManager $blokkliManager
   *   The blokkli manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   */
  public function __construct(
    protected ParagraphsBlokkliManager $blokkliManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected TimeInterface $time,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    $this->logger = $loggerFactory->get('paragraphs_blokkli_scheduler');
  }

  /**
   * Process scheduled edit states and publish them.
   *
   * This method is called during cron runs to find and publish edit states
   * that are scheduled for publishing at or before the current time.
   */
  public function processScheduledEditStates(): void {
    $currentTime = $this->time->getRequestTime();

    // Query for edit states that need to be published.
    $storage = $this->entityTypeManager->getStorage('paragraphs_blokkli_edit_state');
    $query = $storage->getQuery()
      ->condition('publish_on', NULL, 'IS NOT NULL')
      ->condition('publish_on', $currentTime, '<=')
      ->accessCheck(FALSE)
      ->sort('publish_on', 'ASC');

    $ids = $query->execute();

    if (empty($ids)) {
      return;
    }

    $this->logger->info('Processing @count scheduled edit states for publishing.', [
      '@count' => count($ids),
    ]);

    /** @var \Drupal\paragraphs_blokkli\Entity\ParagraphsBlokkliEditState[] $editStates */
    $editStates = $storage->loadMultiple($ids);

    foreach ($editStates as $editState) {
      $this->publishEditState($editState);
    }
  }

  /**
   * Publish a single edit state.
   *
   * @param \Drupal\paragraphs_blokkli\Entity\ParagraphsBlokkliEditState $editState
   *   The edit state to publish.
   */
  protected function publishEditState(EditStateInterface $editState): void {
    try {
      $success = $this->blokkliManager->publish($editState, TRUE, FALSE, TRUE);

      if ($success) {
        $this->logger->info('Successfully published scheduled edit state @id for host entity @entity_type:@entity_uuid.', [
          '@id' => $editState->id(),
          '@entity_type' => $editState->getHostEntityType(),
          '@entity_uuid' => $editState->getHostEntityUuid(),
        ]);
      }
      else {
        $this->logger->error('Failed to publish scheduled edit state @id due to validation errors.', [
          '@id' => $editState->id(),
        ]);
      }
    }
    catch (\Exception $e) {
      $this->logger->critical('Error publishing scheduled edit state @id: @message', [
        '@id' => $editState->id(),
        '@message' => $e->getMessage(),
      ]);
    }
  }

}
