<?php

namespace Drupal\paragraphs_blokkli_scheduler\Plugin\GraphQL\DataProducer;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\paragraphs\ParagraphInterface;
use Drupal\scheduler\SchedulerManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Whether the paragraph bundle supports the given scheduler process.
 *
 * @DataProducer(
 *   id = "paragraphs_blokkli_scheduler_paragraph_schedule_date",
 *   name = @Translation("Paragraphs Blokkli Scheduler Paragraph Schedule Date"),
 *   description = @Translation("The scheduler date value."),
 *   produces = @ContextDefinition("string",
 *     label = @Translation("The date.")
 *   ),
 *   consumes = {
 *     "paragraph" = @ContextDefinition("entity:paragraph",
 *       label = @Translation("The paragraph entity.")
 *     ),
 *     "process" = @ContextDefinition("string",
 *       label = @Translation("The scheduling process."),
 *     ),
 *   }
 * )
 */
class ParagraphScheduleDateValue extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('scheduler.manager')
    );
  }

  /**
   * HasSchedulerField constructor.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $plugin_id
   *   The plugin id.
   * @param array $plugin_definition
   *   The plugin definition array.
   * @param \Drupal\scheduler\SchedulerManager $schedulerManager
   *   The scheduler manager.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    protected SchedulerManager $schedulerManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Resolver.
   *
   * @param \Drupal\paragraphs\Entity\Paragraph $paragraph
   *   The paragraph entity.
   * @param string $process
   *   The scheduling process ('publish' or 'unpublish').
   *
   * @return string|null
   *   The date.
   */
  public function resolve(ParagraphInterface $paragraph, string $process): ?string {
    $fieldName = $process === 'publish' ? 'publish_on' : 'unpublish_on';
    if (!$paragraph->hasField($fieldName)) {
      return NULL;
    }

    $value = $paragraph->get($fieldName)->value;

    if ($value) {
      $dateTime = DrupalDateTime::createFromTimestamp($value, new \DateTimeZone('UTC'));
      return $dateTime->format('Y-m-d\TH:i:s.v\Z');
    }

    return NULL;
  }

}
