<?php

namespace Drupal\paragraphs_blokkli_scheduler\Plugin\GraphQL\SchemaExtension;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\graphql\GraphQL\ResolverBuilder;
use Drupal\graphql\GraphQL\ResolverRegistryInterface;
use Drupal\graphql\Plugin\GraphQL\SchemaExtension\SdlSchemaExtensionPluginBase;
use Drupal\paragraphs\ParagraphInterface;

/**
 * A schema extension to add support for the paragraphs_blokkli_scheduler module.
 *
 * @SchemaExtension(
 *   id = "paragraphs_blokkli_scheduler",
 *   name = "Paragraphs Blokkli Scheduler",
 *   description = "Add support for scheduler module.",
 *   schema = "core_composable"
 * )
 */
class SchedulerExtension extends SdlSchemaExtensionPluginBase {

  /**
   * {@inheritdoc}
   */
  public function registerResolvers(ResolverRegistryInterface $registry): void {
    $builder = new ResolverBuilder();

    $registry->addFieldResolver(
      'Paragraph',
      'isScheduled',
      $builder->callback(function (ParagraphInterface $paragraph) {

        if ($paragraph->hasField('publish_on') && $paragraph->hasField('unpublish_on')) {
          $is_publish_enabled = !$paragraph->get('publish_on')->isEmpty();
          $is_unpublish_enabled = !$paragraph->get('unpublish_on')->isEmpty();
          return ($is_publish_enabled || $is_unpublish_enabled);
        }
        return FALSE;
      })
    );

    $registry->addFieldResolver(
      'ParagraphsType',
      'hasPublishOn',
      $builder->produce('paragraphs_blokkli_scheduler_supports_process')
        ->map('type', $builder->fromParent())
        ->map('process', $builder->fromValue('publish'))
    );

    $registry->addFieldResolver(
      'ParagraphsType',
      'hasUnpublishOn',
      $builder->produce('paragraphs_blokkli_scheduler_supports_process')
        ->map('type', $builder->fromParent())
        ->map('process', $builder->fromValue('unpublish'))
    );

    $registry->addFieldResolver(
      'ParagraphsEditMutationState',
      'unschedule',
      $builder->produce('paragraphs_blokkli_unschedule')
        ->map('state', $builder->fromParent())
    );

    $registry->addFieldResolver(
      'ParagraphsBlokkliParagraphEditContext',
      'publishOn',
      $builder->produce('paragraphs_blokkli_scheduler_paragraph_schedule_date')
        ->map('paragraph', $builder->fromParent())
        ->map('process', $builder->fromValue('publish'))
    );

    $registry->addFieldResolver(
      'ParagraphsBlokkliParagraphEditContext',
      'unpublishOn',
      $builder->produce('paragraphs_blokkli_scheduler_paragraph_schedule_date')
        ->map('paragraph', $builder->fromParent())
        ->map('process', $builder->fromValue('unpublish'))
    );

    $registry->addFieldResolver(
      'ParagraphsEditMutationState',
      'schedule',
      $builder->produce('paragraphs_blokkli_schedule')
        ->map('state', $builder->fromParent())
        ->map('date', $builder->fromArgument('date'))
        ->map('revisionLogMessage', $builder->fromArgument('revisionLogMessage'))
    );

    $registry->addFieldResolver(
      'ParagraphsBlokkliPublishOptions',
      'canSchedule',
      $builder->fromValue(TRUE)
    );

    $registry->addFieldResolver(
      'ParagraphsBlokkliPublishOptions',
      'publishOn',
      $builder->callback(static function (array $options) {
        /** @var \Drupal\paragraphs_blokkli\Entity\ParagraphsBlokkliEditState|null $state */
        $state = $options['state'] ?? NULL;
        if (!$state) {
          return NULL;
        }

        $timestamp = $state->get('publish_on')->first()?->value;
        if (!$timestamp) {
          return NULL;
        }

        $dateTime = DrupalDateTime::createFromTimestamp($timestamp, new \DateTimeZone('UTC'));
        return $dateTime->format('Y-m-d\TH:i:s.v\Z');
      })
    );

    $registry->addFieldResolver(
      'ParagraphsBlokkliPublishOptions',
      'revisionLogMessage',
      $builder->callback(static function (array $options) {
        /** @var \Drupal\paragraphs_blokkli\Entity\ParagraphsBlokkliEditState|null $state */
        $state = $options['state'] ?? NULL;
        if (!$state) {
          return NULL;
        }

        return $state->getRevisionMessage();
      })
    );

  }

}
