<?php

namespace Drupal\paragraphs_blokkli\Model\PluginConfig;

/**
 * Text input configuration.
 */
final class TextInput extends ConfigInputBase {

  /**
   * The minimum length.
   *
   * @var int|null
   */
  protected ?int $minLength = NULL;

  /**
   * The maximum length.
   *
   * @var int|null
   */
  protected ?int $maxLength = NULL;

  /**
   * The placeholder text.
   *
   * @var string
   */
  protected string $placeholder = '';

  /**
   * The validation pattern (regex).
   *
   * @var string|null
   */
  protected ?string $pattern = NULL;

  /**
   * Whether this is a multiline text area.
   *
   * @var bool
   */
  protected bool $multiline = FALSE;

  /**
   * The number of rows for multiline text.
   *
   * @var int
   */
  protected int $rows = 3;

  /**
   * Sets the minimum length.
   *
   * @param int $length
   *   The minimum length.
   *
   * @return $this
   */
  public function minLength(int $length): static {
    $this->minLength = $length;
    return $this;
  }

  /**
   * Sets the maximum length.
   *
   * @param int $length
   *   The maximum length.
   *
   * @return $this
   */
  public function maxLength(int $length): static {
    $this->maxLength = $length;
    return $this;
  }

  /**
   * Sets the placeholder text.
   *
   * @param string $placeholder
   *   The placeholder text.
   *
   * @return $this
   */
  public function placeholder(string $placeholder): static {
    $this->placeholder = $placeholder;
    return $this;
  }

  /**
   * Sets the validation pattern.
   *
   * @param string $pattern
   *   The regex pattern for validation.
   *
   * @return $this
   */
  public function pattern(string $pattern): static {
    $this->pattern = $pattern;
    return $this;
  }

  /**
   * Marks this as a multiline text area.
   *
   * @param int $rows
   *   The number of rows to display.
   *
   * @return $this
   */
  public function multiline(int $rows = 3): static {
    $this->multiline = TRUE;
    $this->rows = $rows;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getType(): string {
    return 'text';
  }

  /**
   * Gets the minimum length.
   *
   * @return int|null
   *   The minimum length or NULL if not set.
   */
  public function getMinLength(): ?int {
    return $this->minLength;
  }

  /**
   * Gets the maximum length.
   *
   * @return int|null
   *   The maximum length or NULL if not set.
   */
  public function getMaxLength(): ?int {
    return $this->maxLength;
  }

  /**
   * Gets the placeholder text.
   *
   * @return string
   *   The placeholder text.
   */
  public function getPlaceholder(): string {
    return $this->placeholder;
  }

  /**
   * Gets the validation pattern.
   *
   * @return string|null
   *   The pattern or NULL if not set.
   */
  public function getPattern(): ?string {
    return $this->pattern;
  }

  /**
   * Checks if this is a multiline text area.
   *
   * @return bool
   *   TRUE if multiline, FALSE otherwise.
   */
  public function isMultiline(): bool {
    return $this->multiline;
  }

  /**
   * Gets the number of rows for multiline text.
   *
   * @return int
   *   The number of rows.
   */
  public function getRows(): int {
    return $this->rows;
  }

  /**
   * {@inheritdoc}
   */
  public function toArray(): array {
    $data = parent::toArray();
    $data['minLength'] = $this->minLength;
    $data['maxLength'] = $this->maxLength;
    $data['placeholder'] = $this->placeholder;
    $data['pattern'] = $this->pattern;
    $data['multiline'] = $this->multiline;
    $data['rows'] = $this->rows;
    return $data;
  }

}
