<?php

namespace Drupal\paragraphs_blokkli_graphql\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\paragraphs_blokkli\ParagraphsBlokkliConfig;
use Drupal\paragraphs_blokkli\Plugin\Field\FieldType\ParagraphsBlokkliHostOptionsFieldItemList;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Build the props for the blokkli provider component.
 *
 * @DataProducer(
 *   id = "paragraphs_blokkli_entity_config",
 *   name = @Translation("Paragraphs Blokkli Entity Config"),
 *   description = @Translation("Returns the blokkli entity config"),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The entity config.")
 *   ),
 *   consumes = {
 *     "entity" = @ContextDefinition("entity",
 *       label = @Translation("The entity.")
 *     ),
 *   }
 * )
 */
class EntityConfig extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user'),
      $container->get('paragraphs_blokkli.config'),
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * EntityConfig constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\paragraphs_blokkli\ParagraphsBlokkliConfig $blokkliConfig
   *   The blokkli config.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfo $bundleInfo
   *   The bundle info service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    protected AccountInterface $currentUser,
    protected ParagraphsBlokkliConfig $blokkliConfig,
    protected EntityTypeBundleInfoInterface $bundleInfo,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Resolver.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $fieldContext
   *   The field context.
   *
   * @return array
   *   The blokkli props.
   */
  public function resolve(EntityInterface $entity, FieldContext $fieldContext): array {
    return [
      'linkPath' => $this->getLinkPath($entity),
    ];
  }

  /**
   * Get the value for the linkPath field.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return string|null
   *   The value.
   */
  private function getLinkPath(EntityInterface $entity): ?string {
    if ($entity->hasLinkTemplate('canonical')) {
      return $entity->toUrl('canonical', [
        'path_processing' => FALSE,
      ])->toString();
    }

    return NULL;
  }

}
