<?php

namespace Drupal\paragraphs_blokkli_graphql\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\RevisionLogInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\paragraphs_blokkli\EditStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns the available publish options.
 *
 * @DataProducer(
 *   id = "paragraphs_blokkli_publish_options",
 *   name = @Translation("Paragraphs Blokkli Publish Options"),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The props.")
 *   ),
 *   consumes = {
 *     "state" = @ContextDefinition("any",
 *       label = @Translation("The edit state.")
 *     ),
 *   }
 * )
 */
class PublishOptions extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * CurrentUser constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfo $bundleInfo
   *   The bundle info service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $bundleInfo,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Resolver.
   *
   * @param \Drupal\paragraphs_blokkli\Entity\ParagraphsBlokkliEditState $state
   *   The edit state.
   *
   * @return array
   *   The blokkli props.
   */
  public function resolve(EditStateInterface $state): array|null {
    $entity = $state->getHostEntity();
    if (!$entity) {
      return NULL;
    }

    $canPublish = $entity->access('update');
    $entityType = $entity->getEntityType();

    $isRevisionable = $entity instanceof RevisionableInterface
      && $entityType->isRevisionable()
      && $entityType->hasKey('revision');

    return [
      'canPublish' => $canPublish,
      'isRevisionable' => $isRevisionable,
      'hasRevisionLogMessage' => $isRevisionable && $entity instanceof RevisionLogInterface,
      'lastChanged' => $entity instanceof EntityChangedInterface ? $entity->getChangedTime() : NULL,
      'state' => $state,
      'revisionLogMessage' => $state->getRevisionMessage(),
    ];
  }

}
