<?php

namespace Drupal\paragraphs_blokkli_scheduler\Plugin\GraphQL\DataProducer;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\paragraphs_blokkli\EditStateInterface;
use Drupal\paragraphs_blokkli\Entity\ParagraphsBlokkliEditState;

/**
 * Unschedule an edit state.
 *
 * @DataProducer(
 *   id = "paragraphs_blokkli_schedule",
 *   name = @Translation("Paragraphs Blokkli Schedule"),
 *   description = @Translation("Schedule an edit state."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("The mutation result.")
 *   ),
 *   consumes = {
 *     "state" = @ContextDefinition("any",
 *       label = @Translation("The paragraphs edit state.")
 *     ),
 *     "date" = @ContextDefinition("string",
 *       label = @Translation("The date."),
 *     ),
 *     "revisionLogMessage" = @ContextDefinition("string",
 *       label = @Translation("The revision log message."),
 *       required = FALSE
 *     ),
 *   }
 * )
 */
class ScheduleEditState extends DataProducerPluginBase {

  /**
   * Resolver.
   *
   * @param \Drupal\paragraphs_blokkli\EditStateInterface $state
   *   The edit state.
   * @param string $date
   *   The date.
   * @param string|null $revisionLogMessage
   *   The revision log message.
   */
  public function resolve(EditStateInterface $state, string $date, ?string $revisionLogMessage) {
    assert($state instanceof ParagraphsBlokkliEditState);

    $timestamp = NULL;

    try {
      $dateTime = new DrupalDateTime($date, new \DateTimeZone('UTC'));
      $timestamp = $dateTime->getTimestamp();
    }
    catch (\Exception $e) {
      return [
        'success' => FALSE,
        'errors' => ['Invalid date'],
      ];
    }

    // Validate that the date is at least 2 minutes in the future.
    $currentTime = \Drupal::time()->getRequestTime();
    $minimumTime = $currentTime + 120;

    if ($timestamp < $minimumTime) {
      return [
        'success' => FALSE,
        'errors' => ['The scheduled date must be at least 2 minutes in the future.'],
      ];
    }

    $state->set('publish_on', $timestamp);
    $state->setRevisionMessage($revisionLogMessage);
    $state->save();

    return [
      'success' => TRUE,
      'state' => $state,
    ];
  }

}
