<?php

namespace Drupal\paragraphs_blokkli_scheduler\Plugin\ParagraphsBlokkli\Mutation;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\paragraphs_blokkli\Exception\MutationException;
use Drupal\paragraphs_blokkli\ParagraphMutationContextInterface;
use Drupal\paragraphs_blokkli\ParagraphMutationPluginBase;
use Drupal\paragraphs_blokkli\ParagraphsBlokkliHelper;
use Drupal\scheduler\SchedulerManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Set the scheduler dates for paragraphs.
 *
 * @ParagraphMutation(
 *   id = "set_paragraph_schedule",
 *   label = @Translation("Set Paragraph Schedule"),
 *   description = @Translation("Set the scheduler dates for paragraphs."),
 *   arguments = {
 *     "items" = @ContextDefinition("map",
 *       label = @Translation("The UUID of the paragraph being edited."),
 *       multiple = TRUE
 *     ),
 *   }
 * )
 */
class SetParagraphSchedule extends ParagraphMutationPluginBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('uuid'),
      $container->get('paragraphs_blokkli.helper'),
      $container->get('scheduler.manager')
    );
  }

  /**
   * AddVideoRemote constructor.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $pluginId
   *   The plugin id.
   * @param array $pluginDefinition
   *   The plugin definition array.
   * @param EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param UuidInterface $uuidHelper
   *   The UUID helper.
   * @param ParagraphsBlokkliHelper $paragraphsBlokkliHelper
   *   The paragraphs blokkli helper.
   * @param \Drupal\scheduler\SchedulerManager $schedulerManager
   *   The scheduler manager.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    array $pluginDefinition,
    EntityTypeManagerInterface $entityTypeManager,
    UuidInterface $uuidHelper,
    ParagraphsBlokkliHelper $paragraphsBlokkliHelper,
    protected SchedulerManager $schedulerManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition, $entityTypeManager, $uuidHelper, $paragraphsBlokkliHelper);
  }

  /**
   * Execute the mutation.
   *
   * @param ParagraphMutationContextInterface $context
   *   The paragraph mutation context.
   * @param array $items
   *   The schedule items.
   */
  public function execute(ParagraphMutationContextInterface $context, array $items): void {
    $publishOnBundles = $this->schedulerManager->getEnabledTypes('paragraph', 'publish');
    $unpublishOnBundles = $this->schedulerManager->getEnabledTypes('paragraph', 'unpublish');

    foreach ($items as $item) {
      $uuid = $item['uuid'];
      $type = $item['type'];
      $date = $item['date'] ?? NULL;

      $proxy = $context->getProxy($uuid);
      if (!$proxy) {
        continue;
      }

      $paragraph = $proxy->getParagraph();
      if (!$paragraph) {
        continue;
      }

      $bundle = $paragraph->bundle();

      $timestamp = NULL;

      if ($date) {
        try {
          $dateTime = new DrupalDateTime($date, new \DateTimeZone('UTC'));
          $timestamp = $dateTime->getTimestamp();
        }
        catch (\Exception $e) {
          throw new MutationException('Invalid date.', 0, $e);
        }
      }

      if ($type === 'publish' && in_array($bundle, $publishOnBundles)) {
        $paragraph->set('publish_on', $timestamp);
        // If publish_on is in the future, unpublish the paragraph.
        if ($timestamp && $timestamp > time()) {
          $paragraph->setUnpublished();
        }
        else {
          $paragraph->setPublished();
        }
      }
      elseif ($type === 'unpublish' && in_array($bundle, $unpublishOnBundles)) {
        $paragraph->set('unpublish_on', $timestamp);
      }
    }
  }

}
