<?php

declare(strict_types=1);

namespace Drupal\paragraphs_blokkli\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\Checkboxes;
use Drupal\image\Entity\ImageStyle;
use Drupal\paragraphs_blokkli\ParagraphsBlokkliConfig;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Paragraph Builder settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * Constructs a \Drupal\system\ConfigFormBase object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfo $bundleInfo
   *   The entity type bundle info.
   * @param \Drupal\Core\Entity\EntityFieldManager $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Entity\EntityTypeManager $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    protected EntityTypeBundleInfoInterface $bundleInfo,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configFactory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.bundle.info'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'paragraphs_blokkli_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['paragraphs_blokkli.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'schema_file' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form[ParagraphsBlokkliConfig::SCHEMA_FILE] = [
      '#type' => 'textfield',
      '#title' => $this->t('blökkli Schema File'),
      '#description' => $this->t('The path of the schema file generated by blökkli (relative to the web root). If set, the paragraph options defined in blökkli are also available as behaviour settings on the paragraphs.'),
      '#default_value' => $this->getConfigValue('schema_file'),
    ];

    $form[ParagraphsBlokkliConfig::HIDE_UNPUBLISHED_PARAGRAPHS] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide unpublished paragraphs'),
      '#description' => $this->t('If checked, unpublished paragraphs will be hidden for all users when viewing the page (but not in edit mode). This overrides the default behavior of showing unpublished content to logged-in users.'),
      '#default_value' => $this->getConfigValue(ParagraphsBlokkliConfig::HIDE_UNPUBLISHED_PARAGRAPHS),
    ];

    $form['#tree'] = TRUE;

    $form['tabs'] = [
      '#type' => 'vertical_tabs',
    ];

    $this->buildValidationsForm($form, $form_state);
    $this->buildImageStyleForm($form, $form_state);
    $this->buildClipboardForm($form, $form_state);
    $this->buildEnabledEntityTypesForm($form, $form_state);
    return parent::buildForm($form, $form_state);
  }

  /**
   * Get a configuration value.
   *
   * @param string $key
   *   The key.
   *
   * @return string|array|null
   *   The value.
   */
  private function getConfigValue(string $key) {
    return $this->config('paragraphs_blokkli.settings')->get($key);
  }

  /**
   * Build the form to configure the enabled entity types and bundles.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormState $form_state
   *   The form state.
   */
  private function buildEnabledEntityTypesForm(array &$form, FormStateInterface $form_state) {
    $definitions = $this->entityTypeManager->getDefinitions();

    $form['enabled_types'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#collapsible' => FALSE,
      '#title' => $this->t('Enabled entity types'),
      '#description' => $this->t('Select for which entity types and bundles blökkli should be enabled.'),
    ];

    $selected = $this->getConfigValue(ParagraphsBlokkliConfig::ENABLED_TYPES) ?? [];

    foreach ($definitions as $entityTypeId => $definition) {
      // Paragraphs by design can not be edit state hosts.
      if ($entityTypeId === 'paragraph') {
        continue;
      }

      if ($definition instanceof ContentEntityTypeInterface) {
        $label = $definition->getLabel();
        $options[$entityTypeId] = "$label ($entityTypeId)";
        $bundleInfo = $this->bundleInfo->getBundleInfo($entityTypeId);

        // We only support entity types that have bundles.
        if (count($bundleInfo) === 1 && array_keys($bundleInfo)[0] === $entityTypeId) {
          continue;
        }

        $options = [];

        foreach ($bundleInfo as $bundle => $info) {
          $bundleLabel = $info['label'];
          $options[$bundle] = "$bundleLabel ($bundle)";
        }

        $selectedBundles = $selected[$entityTypeId] ?? [];

        $form['enabled_types'][$entityTypeId] = [
          '#type' => 'checkboxes',
          '#title' => "$label ($entityTypeId)",
          '#options' => $options,
          '#default_value' => $selectedBundles,
        ];
      }
    }
  }

  /**
   * Build the clipboard form.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormState $form_state
   *   The form state.
   */
  private function buildClipboardForm(array &$form, FormStateInterface $form_state) {
    $form['clipboard'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#collapsible' => FALSE,
      '#title' => $this->t('Clipboard Mapping'),
      '#description' => $this->t('Define which entities to create when pasting clipboard content in blökkli.'),
    ];
    $form['clipboard'][ParagraphsBlokkliConfig::MEDIA_BUNDLE_REMOTE_VIDEO] = [
      '#type' => 'select',
      '#title' => $this->t('Remote Video Media Bundle'),
      '#description' => $this->t('Select the media bundle for remote videos (such as YouTube or Vimeo). If a bundle is selected, pasting a supported remote video link in blökkli will create a media entity of the given bundle and add a matching paragraph that references the media entity.'),
      '#default_value' => $this->getConfigValue(ParagraphsBlokkliConfig::MEDIA_BUNDLE_REMOTE_VIDEO),
      '#options' => $this->getMediaSourceOptions('oembed:video'),
    ];

    $form['clipboard'][ParagraphsBlokkliConfig::MEDIA_BUNDLE_IMAGE] = [
      '#type' => 'select',
      '#title' => $this->t('Image Media Bundle'),
      '#description' => $this->t('Select the media bundle for images. If a bundle is selected, pasting an image in blökkli will create a media entity of the given bundle and add a matching paragraph that references the media entity.'),
      '#default_value' => $this->getConfigValue(ParagraphsBlokkliConfig::MEDIA_BUNDLE_IMAGE),
      '#options' => $this->getMediaSourceOptions('image'),
    ];

    $form['clipboard'][ParagraphsBlokkliConfig::MEDIA_BUNDLE_FILE] = [
      '#type' => 'select',
      '#title' => $this->t('File Media Bundle'),
      '#description' => $this->t('Select the media bundle for files. If a bundle is selected, pasting a file in blökkli will create a media entity of the given bundle and add a matching paragraph that references the media entity.'),
      '#default_value' => $this->getConfigValue(ParagraphsBlokkliConfig::MEDIA_BUNDLE_FILE),
      '#options' => $this->getMediaSourceOptions('file'),
    ];

    $clipboardTextSetting = $this->getConfigValue(ParagraphsBlokkliConfig::CLIPBOARD_TEXT_PARAGRAPH);
    $clipboardTextKey = '';

    if (is_array($clipboardTextSetting) && !empty($clipboardTextSetting['bundle']) && !empty($clipboardTextSetting['field'])) {
      $bundle = $clipboardTextSetting['bundle'];
      $field = $clipboardTextSetting['field'];
      $clipboardTextKey = "$bundle:$field";
    }

    $form['clipboard'][ParagraphsBlokkliConfig::CLIPBOARD_TEXT_PARAGRAPH] = [
      '#type' => 'select',
      '#title' => $this->t('Clipboard Text Paragraph'),
      '#description' => $this->t('Select the paragraph and field for rich text. If selected, pasting rich text from the clipboard in blökkli will create a new paragraph of the given bundle and add the pasted text in the given field.'),
      '#default_value' => $clipboardTextKey,
      '#options' => $this->getClipboardTextOptions(),
    ];
  }

  /**
   * Build the form for validation settings.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormState $form_state
   *   The form state.
   */
  private function buildValidationsForm(array &$form, FormStateInterface $form_state) {
    $selected = $this->getConfigValue(ParagraphsBlokkliConfig::VALIDATION_TYPES) ?? [];

    $form[ParagraphsBlokkliConfig::VALIDATION_TYPES] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Validations'),
      '#description' => $this->t('Select which types of validations to perform during editing and publishing.'),
      '#options' => [
        'host' => $this->t('Host Entity'),
        'field' => $this->t('Paragraph Fields'),
        'paragraph' => $this->t('Paragraph'),
      ],
      '#default_value' => $selected,
      'host' => [
        '#description' => $this->t('Performs validations on the host entity (like node).'),
      ],
      'field' => [
        '#description' => $this->t('Validate on the entity_reference_revisions fields containing paragraphs.'),
      ],
      'paragraph' => [
        '#description' => $this->t('Validates each existing or mutated paragraph.'),
      ],
    ];
  }

  /**
   * Build the image style form.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormState $form_state
   *   The form state.
   */
  private function buildImageStyleForm(array &$form, FormStateInterface $form_state) {
    $imageStyles = [];

    foreach (ImageStyle::loadMultiple() as $style) {
      $imageStyles[$style->id()] = $style->label();
    }

    $form[ParagraphsBlokkliConfig::IMAGE_STYLE] = [
      '#type' => 'select',
      '#title' => $this->t('Image style'),
      '#description' => $this->t('The image style used for the media library and content search.'),
      '#default_value' => $this->getConfigValue(ParagraphsBlokkliConfig::IMAGE_STYLE),
      '#options' => $imageStyles,
    ];
  }

  /**
   * Get the possible options for the clipboard text setting.
   *
   * @return array
   *   The options.
   */
  private function getClipboardTextOptions(): array {
    $options = [
      '' => $this->t('Select a bundle and field'),
    ];

    $bundles = $this->bundleInfo->getBundleInfo('paragraph');

    foreach ($bundles as $bundle => $bundleInfo) {
      $fields = $this->entityFieldManager->getFieldDefinitions('paragraph', $bundle);

      foreach ($fields as $fieldName => $field) {
        if ($field->getType() === 'text_long') {
          $key = "$bundle:$fieldName";
          $fieldLabel = $field->getLabel();
          $bundleLabel = $bundleInfo['label'];
          $options[$key] = "$bundleLabel - $fieldName ($fieldLabel)";
        }
      }
    }

    return $options;
  }

  /**
   * Get supported media bundle options for the given source.
   *
   * @param string $source
   *   The media source ID.
   *
   * @return array
   *   The options.
   */
  private function getMediaSourceOptions(string $source): array {
    $mediaStorage = $this->entityTypeManager->getStorage('media_type');

    // Get media types that support remote videos.
    $bundles = array_values(
      $mediaStorage
        ->getQuery()
        ->accessCheck(FALSE)
        ->condition('source', $source)
        ->execute()
    );
    $bundleInfo = $this->bundleInfo->getBundleInfo('media');

    $options = [
      '' => $this->t('Select a media type'),
    ];

    foreach ($bundles as $bundle) {
      $info = $bundleInfo[$bundle];
      $options[$bundle] = $info['label'];
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $file_path = $form_state->getValue(ParagraphsBlokkliConfig::SCHEMA_FILE);
    if (!empty($file_path) && !file_exists($file_path)) {
      $form_state->setErrorByName('schema_file', $this->t('The file does not exist.'));
    }

    /** @var \Drupal\paragraphs_blokkli\PbEntityMappingStorage $mappingStorage */
    $mappingStorage = $this->entityTypeManager->getStorage('pb_entity_mapping');

    $keysToValidate = [
      ParagraphsBlokkliConfig::MEDIA_BUNDLE_REMOTE_VIDEO,
      ParagraphsBlokkliConfig::MEDIA_BUNDLE_FILE,
      ParagraphsBlokkliConfig::MEDIA_BUNDLE_IMAGE,
    ];

    foreach ($keysToValidate as $key) {
      $bundle = $form_state->getValue(['clipboard', $key]);
      if ($bundle) {
        $mappings = $mappingStorage->findMappings('media', $bundle);

        // Add a warning if no mapping is found for the configured media bundle.
        if (empty($mappings)) {
          $error = $this->t("Missing entity mapping for type '@entity_type' and bundle '@bundle'. A mapping is required to be able to create a paragraph from clipboard content.", [
            '@entity_type' => 'media',
            '@bundle' => $bundle,
          ]);
          $form_state->setError($form['clipboard'][$key], $error);
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('paragraphs_blokkli.settings');
    $config->set(ParagraphsBlokkliConfig::SCHEMA_FILE, $form_state->getValue('schema_file'));
    $config->set(ParagraphsBlokkliConfig::HIDE_UNPUBLISHED_PARAGRAPHS, (bool) $form_state->getValue(ParagraphsBlokkliConfig::HIDE_UNPUBLISHED_PARAGRAPHS));

    $keys = [
      ParagraphsBlokkliConfig::MEDIA_BUNDLE_IMAGE,
      ParagraphsBlokkliConfig::MEDIA_BUNDLE_REMOTE_VIDEO,
      ParagraphsBlokkliConfig::MEDIA_BUNDLE_FILE,
    ];

    foreach ($keys as $key) {
      $config->set($key, $form_state->getValue(['clipboard', $key]));
    }

    $config->set(ParagraphsBlokkliConfig::IMAGE_STYLE, $form_state->getValue('image_style'));

    $clipboardTextKey = $form_state->getValue(['clipboard', ParagraphsBlokkliConfig::CLIPBOARD_TEXT_PARAGRAPH]);
    $clipboardTextSetting = [];

    if (is_string($clipboardTextKey)) {
      [$bundle, $field] = explode(':', $clipboardTextKey);

      if ($bundle && $field) {
        $clipboardTextSetting['bundle'] = $bundle;
        $clipboardTextSetting['field'] = $field;
      }
    }

    $enabledTypesValues = $form_state->getValue('enabled_types');

    $enabledTypes = [];

    foreach ($enabledTypesValues as $entityType => $bundles) {
      $selectedBundles = Checkboxes::getCheckedCheckboxes($bundles);
      if (!empty($selectedBundles)) {
        $enabledTypes[$entityType] = $selectedBundles;
      }
    }

    $config->set(ParagraphsBlokkliConfig::CLIPBOARD_TEXT_PARAGRAPH, $clipboardTextSetting);
    $config->set(ParagraphsBlokkliConfig::ENABLED_TYPES, $enabledTypes);

    $enabledValidations = Checkboxes::getCheckedCheckboxes($form_state->getValue(ParagraphsBlokkliConfig::VALIDATION_TYPES));
    $config->set(ParagraphsBlokkliConfig::VALIDATION_TYPES, array_values($enabledValidations));

    $config->save();

    parent::submitForm($form, $form_state);
  }

}
