<?php

namespace Drupal\paragraphs_blokkli\Model\PluginConfig;

/**
 * Base class for all configuration input types.
 */
abstract class ConfigInputBase {

  /**
   * The machine name of the input.
   *
   * @var string
   */
  protected string $name;

  /**
   * The human-readable label.
   *
   * @var string
   */
  protected string $label = '';

  /**
   * The description text.
   *
   * @var string
   */
  protected string $description = '';

  /**
   * Whether this input is required.
   *
   * @var bool
   */
  protected bool $required = FALSE;

  /**
   * The default value.
   *
   * @var mixed
   */
  protected mixed $defaultValue = NULL;

  /**
   * Constructs a ConfigInputBase instance.
   *
   * @param string $name
   *   The machine name of the input.
   */
  public function __construct(string $name) {
    $this->name = $name;
  }

  /**
   * Sets the label.
   *
   * @param string $label
   *   The human-readable label.
   *
   * @return $this
   */
  public function label(string $label): static {
    $this->label = $label;
    return $this;
  }

  /**
   * Sets the description.
   *
   * @param string $description
   *   The description text.
   *
   * @return $this
   */
  public function description(string $description): static {
    $this->description = $description;
    return $this;
  }

  /**
   * Marks this input as required.
   *
   * @return $this
   */
  public function required(): static {
    $this->required = TRUE;
    return $this;
  }

  /**
   * Sets the default value.
   *
   * @param mixed $value
   *   The default value.
   *
   * @return $this
   */
  public function default(mixed $value): static {
    $this->defaultValue = $value;
    return $this;
  }

  /**
   * Gets the machine name.
   *
   * @return string
   *   The machine name.
   */
  public function getName(): string {
    return $this->name;
  }

  /**
   * Gets the label.
   *
   * @return string
   *   The label.
   */
  public function getLabel(): string {
    return $this->label;
  }

  /**
   * Gets the description.
   *
   * @return string
   *   The description.
   */
  public function getDescription(): string {
    return $this->description;
  }

  /**
   * Checks if this input is required.
   *
   * @return bool
   *   TRUE if required, FALSE otherwise.
   */
  public function isRequired(): bool {
    return $this->required;
  }

  /**
   * Gets the default value.
   *
   * @return mixed
   *   The default value.
   */
  public function getDefaultValue(): mixed {
    return $this->defaultValue;
  }

  /**
   * Returns the input type.
   *
   * @return string
   *   The input type identifier.
   */
  abstract public function getType(): string;

  /**
   * Converts the input to an array representation.
   *
   * @return array<string, mixed>
   *   The array representation.
   */
  public function toArray(): array {
    return [
      'type' => $this->getType(),
      'name' => $this->name,
      'label' => $this->label,
      'description' => $this->description,
      'required' => $this->required,
      'default' => $this->defaultValue,
    ];
  }

}
