<?php

namespace Drupal\paragraphs_blokkli\Model\PluginConfig;

/**
 * Options input configuration (radio/select).
 */
final class OptionsInput extends ConfigInputBase {

  /**
   * The variant type (radio or select).
   *
   * @var string
   */
  protected string $variant = 'select';

  /**
   * The available options.
   *
   * @var array<string, string>
   */
  protected array $options = [];

  /**
   * Sets the variant type.
   *
   * @param string $variant
   *   The variant ('radio' or 'select').
   *
   * @return $this
   *
   * @throws \InvalidArgumentException
   *   When an invalid variant is provided.
   */
  public function variant(string $variant): static {
    if (!in_array($variant, ['radio', 'select'], TRUE)) {
      throw new \InvalidArgumentException("Invalid variant '$variant'. Must be 'radio' or 'select'.");
    }
    $this->variant = $variant;
    return $this;
  }

  /**
   * Adds an option.
   *
   * @param string $value
   *   The option value.
   * @param string $label
   *   The option label.
   *
   * @return $this
   */
  public function option(string $value, string $label): static {
    $this->options[$value] = $label;
    return $this;
  }

  /**
   * Sets all options at once.
   *
   * @param array<string, string> $options
   *   An associative array of value => label pairs.
   *
   * @return $this
   */
  public function options(array $options): static {
    $this->options = $options;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getType(): string {
    return 'options';
  }

  /**
   * Gets the variant type.
   *
   * @return string
   *   The variant ('radio' or 'select').
   */
  public function getVariant(): string {
    return $this->variant;
  }

  /**
   * Gets the options.
   *
   * @return array<string, string>
   *   The options array.
   */
  public function getOptions(): array {
    return $this->options;
  }

  /**
   * {@inheritdoc}
   */
  public function toArray(): array {
    $data = parent::toArray();
    $data['variant'] = $this->variant;
    $options = [];
    foreach ($this->options as $value => $label) {
      $options[] = [
        'value' => $value,
        'label' => $label,
      ];
    }
    $data['options'] = $options;
    return $data;
  }

}
