<?php

namespace Drupal\paragraphs_blokkli\Model\PluginConfig;

/**
 * Build configuration for a plugin.
 */
final class PluginConfigBuilder {

  /**
   * The configuration inputs.
   *
   * @var array<string, ConfigInputBase>
   */
  protected array $inputs = [];

  /**
   * Adds a text input.
   *
   * @param string $name
   *   The machine name of the input.
   *
   * @return \Drupal\paragraphs_blokkli\Model\PluginConfig\TextInput
   *   The text input instance for chaining.
   */
  public function addText(string $name): TextInput {
    $input = new TextInput($name);
    $this->inputs[$name] = $input;
    return $input;
  }

  /**
   * Adds a seed input.
   *
   * @param string $name
   *   The machine name of the input.
   *
   * @return \Drupal\paragraphs_blokkli\Model\PluginConfig\SeedInput
   *   The seed input instance for chaining.
   */
  public function addSeed(string $name): SeedInput {
    $input = new SeedInput($name);
    $this->inputs[$name] = $input;
    return $input;
  }

  /**
   * Adds an options input (radio or select).
   *
   * @param string $name
   *   The machine name of the input.
   *
   * @return \Drupal\paragraphs_blokkli\Model\PluginConfig\OptionsInput
   *   The options input instance for chaining.
   */
  public function addOptions(string $name): OptionsInput {
    $input = new OptionsInput($name);
    $this->inputs[$name] = $input;
    return $input;
  }

  /**
   * Adds a checkbox input.
   *
   * @param string $name
   *   The machine name of the input.
   *
   * @return \Drupal\paragraphs_blokkli\Model\PluginConfig\CheckboxInput
   *   The checkbox input instance for chaining.
   */
  public function addCheckbox(string $name): CheckboxInput {
    $input = new CheckboxInput($name);
    $this->inputs[$name] = $input;
    return $input;
  }

  /**
   * Gets all configuration inputs.
   *
   * @return ConfigInputBase[]
   *   An array of configuration inputs.
   */
  public function getInputs(): array {
    return array_values($this->inputs);
  }

  /**
   * Gets a specific input by name.
   *
   * @param string $name
   *   The machine name of the input.
   *
   * @return \Drupal\paragraphs_blokkli\Model\PluginConfig\ConfigInputBase|null
   *   The input instance or NULL if not found.
   */
  public function getInput(string $name): ?ConfigInputBase {
    return $this->inputs[$name] ?? NULL;
  }

  /**
   * Checks if an input exists.
   *
   * @param string $name
   *   The machine name of the input.
   *
   * @return bool
   *   TRUE if the input exists, FALSE otherwise.
   */
  public function hasInput(string $name): bool {
    return isset($this->inputs[$name]);
  }

  /**
   * Removes an input.
   *
   * @param string $name
   *   The machine name of the input to remove.
   *
   * @return $this
   */
  public function removeInput(string $name): static {
    unset($this->inputs[$name]);
    return $this;
  }

  /**
   * Converts all inputs to an array representation.
   *
   * @return array
   *   An array of input configurations.
   */
  public function toArray(): array {
    $data = [];
    foreach ($this->getInputs() as $input) {
      $data[] = $input->toArray();
    }
    return $data;
  }

  /**
   * Validates configuration values against the defined inputs.
   *
   * @param array<string, mixed> $values
   *   The values to validate.
   *
   * @return array<string, string>
   *   An array of validation errors keyed by field name.
   */
  public function validate(array $values): array {
    $errors = [];

    foreach ($this->inputs as $name => $input) {
      $value = $values[$name] ?? NULL;

      // Check required fields.
      if ($input->isRequired() && ($value === NULL || $value === '')) {
        $errors[$name] = sprintf('The field "%s" is required.', $input->getLabel() ?: $name);
        continue;
      }

      // Type-specific validation.
      if ($value !== NULL && $value !== '') {
        if ($input instanceof TextInput) {
          $this->validateTextInput($input, $value, $errors);
        }
        elseif ($input instanceof OptionsInput) {
          $this->validateOptionsInput($input, $value, $errors);
        }
        elseif ($input instanceof CheckboxInput) {
          $this->validateCheckboxInput($input, $value, $errors);
        }
      }
    }

    return $errors;
  }

  /**
   * Validates a text input value.
   *
   * @param \Drupal\paragraphs_blokkli\Model\PluginConfig\TextInput $input
   *   The text input configuration.
   * @param mixed $value
   *   The value to validate.
   * @param array<string, string> $errors
   *   The errors array to populate.
   */
  protected function validateTextInput(TextInput $input, mixed $value, array &$errors): void {
    if (!is_string($value)) {
      $errors[$input->getName()] = sprintf('The field "%s" must be a string.', $input->getLabel() ?: $input->getName());
      return;
    }

    $length = mb_strlen($value);

    if ($input->getMinLength() !== NULL && $length < $input->getMinLength()) {
      $errors[$input->getName()] = sprintf(
        'The field "%s" must be at least %d characters long.',
        $input->getLabel() ?: $input->getName(),
        $input->getMinLength()
      );
    }

    if ($input->getMaxLength() !== NULL && $length > $input->getMaxLength()) {
      $errors[$input->getName()] = sprintf(
        'The field "%s" must not exceed %d characters.',
        $input->getLabel() ?: $input->getName(),
        $input->getMaxLength()
      );
    }

    if ($input->getPattern() !== NULL && !preg_match($input->getPattern(), $value)) {
      $errors[$input->getName()] = sprintf(
        'The field "%s" does not match the required format.',
        $input->getLabel() ?: $input->getName()
      );
    }
  }

  /**
   * Validates an options input value.
   *
   * @param \Drupal\paragraphs_blokkli\Model\PluginConfig\OptionsInput $input
   *   The options input configuration.
   * @param mixed $value
   *   The value to validate.
   * @param array<string, string> $errors
   *   The errors array to populate.
   */
  protected function validateOptionsInput(OptionsInput $input, mixed $value, array &$errors): void {
    $options = array_keys($input->getOptions());

    // If ($input->isMultiple()) {
    //   if (!is_array($value)) {
    //     $errors[$input->getName()] = sprintf(
    //       'The field "%s" must be an array.',
    //       $input->getLabel() ?: $input->getName()
    //     );
    //     return;
    //   }
    //
    //   foreach ($value as $val) {
    //     if (!in_array($val, $options, TRUE)) {
    //       $errors[$input->getName()] = sprintf(
    //         'The field "%s" contains an invalid option: %s.',
    //         $input->getLabel() ?: $input->getName(),
    //         $val
    //       );
    //       break;
    //     }
    //   }
    // }
    // else {
    if (!is_string($value) && !is_int($value)) {
      $errors[$input->getName()] = sprintf(
        'The field "%s" must be a string or integer.',
        $input->getLabel() ?: $input->getName()
      );
      return;
    }

    if (!in_array((string) $value, $options, TRUE)) {
      $errors[$input->getName()] = sprintf(
        'The field "%s" contains an invalid option: %s.',
        $input->getLabel() ?: $input->getName(),
        $value
      );
    }
    // }
  }

  /**
   * Validates a checkbox input value.
   *
   * @param \Drupal\paragraphs_blokkli\Model\PluginConfig\CheckboxInput $input
   *   The checkbox input configuration.
   * @param mixed $value
   *   The value to validate.
   * @param array<string, string> $errors
   *   The errors array to populate.
   */
  protected function validateCheckboxInput(CheckboxInput $input, mixed $value, array &$errors): void {
    if (!is_bool($value) && $value !== 0 && $value !== 1 && $value !== '0' && $value !== '1') {
      $errors[$input->getName()] = sprintf(
        'The field "%s" must be a boolean value.',
        $input->getLabel() ?: $input->getName()
      );
    }
  }

}
