<?php

namespace Drupal\layout_paragraphs_clipboard\Controller;

use Drupal\Core\Ajax\AjaxHelperTrait;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\layout_paragraphs\LayoutParagraphsLayout;
use Drupal\layout_paragraphs\LayoutParagraphsLayoutRefreshTrait;
use Drupal\layout_paragraphs\LayoutParagraphsLayoutTempstoreRepository;
use Drupal\paragraphs_clipboard\ParagraphsClipboardService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class CopyClipboardController.
 *
 * Copy and Paste a component of a Layout Paragraphs Layout.
 */
class CopyClipboardController extends ControllerBase {

  use LayoutParagraphsLayoutRefreshTrait;
  use AjaxHelperTrait;

  /**
   * The tempstore service.
   *
   * @var \Drupal\layout_paragraphs\LayoutParagraphsLayoutTempstoreRepository
   */
  protected $tempstore;

  /**
   * The clipboard service.
   *
   * @var \Drupal\paragraphs_clipboard\ParagraphsClipboardService
   */
  protected $clipboardService;

  /**
   * The entity repository service.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * Constructs a new CopyClipboardController object.
   *
   * @param \Drupal\layout_paragraphs\LayoutParagraphsLayoutTempstoreRepository $tempstore
   *   The layout tempstore repository.
   * @param \Drupal\paragraphs_clipboard\ParagraphsClipboardService $clipboard_service
   *   The clipboard service.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   */
  public function __construct(LayoutParagraphsLayoutTempstoreRepository $tempstore, ParagraphsClipboardService $clipboard_service, EntityRepositoryInterface $entity_repository) {
    $this->tempstore = $tempstore;
    $this->clipboardService = $clipboard_service;
    $this->entityRepository = $entity_repository;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('layout_paragraphs.tempstore_repository'),
      $container->get('paragraphs_clipboard.paragraphs_clipboard_service'),
      $container->get('entity.repository'),
    );
  }

  /**
   * Duplicates a component and returns appropriate response.
   *
   * @param \Drupal\layout_paragraphs\LayoutParagraphsLayout $layout_paragraphs_layout
   *   The layout paragraphs layout object.
   * @param string $source_uuid
   *   The source component to be cloned.
   *
   * @return array|\Drupal\Core\Ajax\AjaxResponse
   *   A build array or Ajax response.
   */
  public function copyClipboard(LayoutParagraphsLayout $layout_paragraphs_layout, string $source_uuid) {
    $this->setLayoutParagraphsLayout($layout_paragraphs_layout);

    $paragraph = $this->entityRepository->loadEntityByUuid('paragraph', $source_uuid);
    if (!$paragraph || !$paragraph->id()) {
      $response = new AjaxResponse();
      $response->addCommand(new MessageCommand($this->t('Cannot copy an unsaved paragraph.'), NULL, ['error']));
      return $response;
    }

    $clipboard_data = [
      'paragraph_uuid' => $source_uuid,
    ];
    $this->clipboardService->setClipboardData(ParagraphsClipboardService::CLIPBOARD_UUID_KEY, $clipboard_data);

    if ($this->isAjax()) {
      $response = new AjaxResponse();
      if ($this->needsRefresh()) {
        return $this->refreshLayout($response);
      }

      $response->addCommand(new MessageCommand('Paragraph has been copied to clipboard.', NULL, [], FALSE));

      return $response;
    }
    return [
      '#type' => 'layout_paragraphs_builder',
      '#layout_paragraphs_layout' => $layout_paragraphs_layout,
    ];
  }

}
