<?php

namespace Drupal\layout_paragraphs_clipboard\Controller;

use Drupal\Core\Ajax\AfterCommand;
use Drupal\Core\Ajax\AjaxHelperTrait;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\AppendCommand;
use Drupal\Core\Ajax\BeforeCommand;
use Drupal\Core\Ajax\CloseDialogCommand;
use Drupal\Core\Ajax\PrependCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\layout_paragraphs\Ajax\LayoutParagraphsEventCommand;
use Drupal\layout_paragraphs\LayoutParagraphsComponent;
use Drupal\layout_paragraphs\LayoutParagraphsLayout;
use Drupal\layout_paragraphs\LayoutParagraphsLayoutRefreshTrait;
use Drupal\layout_paragraphs\LayoutParagraphsLayoutTempstoreRepository;
use Drupal\layout_paragraphs\Utility\Dialog;
use Drupal\layout_paragraphs_clipboard\PasteClipboardTrait;
use Drupal\paragraphs_clipboard\Access\ParagraphsClipboardAccess;
use Drupal\paragraphs_clipboard\ParagraphsClipboardService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class PasteClipboardController.
 *
 * Copy and Paste a component of a Layout Paragraphs Layout.
 */
class PasteClipboardController extends ControllerBase {

  use LayoutParagraphsLayoutRefreshTrait;
  use AjaxHelperTrait;
  use PasteClipboardTrait;

  /**
   * The tempstore service.
   *
   * @var \Drupal\layout_paragraphs\LayoutParagraphsLayoutTempstoreRepository
   */
  protected $tempstore;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a new PasteClipboardController object.
   *
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\layout_paragraphs\LayoutParagraphsLayoutTempstoreRepository $tempstore
   *   The layout tempstore repository.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   * @param \Drupal\paragraphs_clipboard\ParagraphsClipboardService $clipboard_service
   *   The clipboard service.
   * @param \Drupal\paragraphs_clipboard\Access\ParagraphsClipboardAccess $clipboard_access
   *   The clipboard access service.
   */
  public function __construct(
    EntityRepositoryInterface $entity_repository,
    LayoutParagraphsLayoutTempstoreRepository $tempstore,
    RequestStack $request_stack,
    ParagraphsClipboardService $clipboard_service,
    ParagraphsClipboardAccess $clipboard_access,
  ) {
    $this->entityRepository = $entity_repository;
    $this->tempstore = $tempstore;
    $this->requestStack = $request_stack;
    $this->clipboardService = $clipboard_service;
    $this->clipboardAccess = $clipboard_access;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity.repository'),
      $container->get('layout_paragraphs.tempstore_repository'),
      $container->get('request_stack'),
      $container->get('paragraphs_clipboard.paragraphs_clipboard_service'),
      $container->get('paragraphs_clipboard.paragraphs_clipboard_access')
    );
  }

  /**
   * Duplicates a component and returns appropriate response.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param \Drupal\layout_paragraphs\LayoutParagraphsLayout $layout_paragraphs_layout
   *   The layout paragraphs layout object.
   *
   * @return array|\Drupal\Core\Ajax\AjaxResponse|void
   *   Void or Ajax response.
   */
  public function pasteClipboard(Request $request, LayoutParagraphsLayout $layout_paragraphs_layout) {
    $this->setLayoutParagraphsLayout($layout_paragraphs_layout);
    $paragraph = $this->getParagraphFromClipboard(ParagraphsClipboardService::CLIPBOARD_UUID_KEY);
    if (!$paragraph) {
      return $this->displayErrorMessage($this->t('No paragraph found in the clipboard.'));
    }

    $field_definition = $layout_paragraphs_layout->getParagraphsReferenceField()->getFieldDefinition();
    $real_item_count = $layout_paragraphs_layout->getParagraphsReferenceField()->count();
    $cardinality = $field_definition->getFieldStorageDefinition()->getCardinality();

    $access_result = $this->clipboardAccess->access($paragraph, $field_definition);
    if (!$this->clipboardService->checkCardinality($real_item_count, $cardinality)) {
      return $this->displayErrorMessage($this->t('This field cannot hold more than @count values.', ['@count' => $cardinality]));
    }

    if ($access_result->isForbidden()) {
      return $this->displayErrorMessage($this->t('Access denied.'));
    }

    $source_component = $this->layoutParagraphsLayout->getComponent($paragraph);
    $cloned_paragraph = $source_component->getEntity()->createDuplicate();

    $sibling_uuid = $request->query->get('sibling_uuid');
    $parent_uuid = $request->query->get('parent_uuid');
    $region = $request->query->get('region');
    $placement = $request->query->get('placement');

    if ($sibling_uuid && $placement) {
      switch ($placement) {
        case 'before':
          $this->layoutParagraphsLayout->insertBeforeComponent($sibling_uuid, $cloned_paragraph);
          break;

        case 'after':
          $this->layoutParagraphsLayout->insertAfterComponent($sibling_uuid, $cloned_paragraph);
          break;
      }
    }
    elseif ($parent_uuid && $region) {
      $this->layoutParagraphsLayout->insertIntoRegion($parent_uuid, $region, $cloned_paragraph);
    }
    else {
      $this->layoutParagraphsLayout->appendComponent($cloned_paragraph);
    }

    if ($source_component->isLayout()) {
      $section = $this->layoutParagraphsLayout->getLayoutSection($source_component->getEntity());
      foreach ($section->getComponents() as $component) {
        $this->layoutParagraphsLayout->duplicateComponent($component->getEntity()->uuid(), $cloned_paragraph->uuid());
      }
    }

    $this->tempstore->set($this->layoutParagraphsLayout);
    $duplicate_component = new LayoutParagraphsComponent($cloned_paragraph);
    if ($this->isAjax()) {
      $response = new AjaxResponse();

      $uuid = $duplicate_component->getEntity()->uuid();
      $rendered_item = [
        '#type' => 'layout_paragraphs_builder',
        '#layout_paragraphs_layout' => $this->layoutParagraphsLayout,
        '#uuid' => $uuid,
      ];

      $selector = Dialog::dialogSelector($this->layoutParagraphsLayout);
      $response->addCommand(new CloseDialogCommand($selector));

      if ($this->needsRefresh()) {
        return $this->refreshLayout($response);
      }

      if ($sibling_uuid && $placement) {
        $selector = '[data-uuid="' . $sibling_uuid . '"]';
      }
      elseif ($parent_uuid && $region) {
        $selector = '[data-region-uuid="' . $parent_uuid . '-' . $region . '"]';
      }
      else {
        $selector = '[data-lpb-id="' . $this->layoutParagraphsLayout->id() . '"]';
      }

      switch ($placement) {
        case 'before':
          $response->addCommand(new BeforeCommand($selector, $rendered_item));
          break;

        case 'after':
          $response->addCommand(new AfterCommand($selector, $rendered_item));
          break;

        case 'append':
          $response->addCommand(new AppendCommand($selector, $rendered_item));
          break;

        case 'prepend':
          $response->addCommand(new PrependCommand($selector, $rendered_item));
          break;
      }

      $response->addCommand(new LayoutParagraphsEventCommand($this->layoutParagraphsLayout, $uuid, 'component:insert'));

      return $response;
    }

    return $this->displayErrorMessage($this->t('Something went wrong.'));
  }

}
