<?php

namespace Drupal\layout_paragraphs_clipboard\EventSubscriber;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Drupal\layout_paragraphs\Event\LayoutParagraphsAllowedTypesEvent;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs_clipboard\Access\ParagraphsClipboardAccess;
use Drupal\paragraphs_clipboard\ParagraphsClipboardService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class AllowedTypesSubscriber.
 *
 * Subscribes to the LayoutParagraphsAllowedTypesEvent to alter component types.
 */
class LayoutParagraphsClipboardAllowedTypesSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * Paragraphs Clipboard Service.
   *
   * @var \Drupal\paragraphs_clipboard\ParagraphsClipboardService
   */
  protected $paragraphsClipboardService;

  /**
   * Paragraphs Clipboard Access.
   *
   * @var \Drupal\paragraphs_clipboard\Access\ParagraphsClipboardAccess
   */
  protected $paragraphsClipboardAccess;

  /**
   * Entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The temp store factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * Constructs a new LayoutParagraphsClipboardAllowedTypesSubscriber.
   *
   * @param \Drupal\paragraphs_clipboard\ParagraphsClipboardService $paragraphs_clipboard_service
   *   The clipboard service.
   * @param \Drupal\paragraphs_clipboard\Access\ParagraphsClipboardAccess $paragraphs_clipboard_access
   *   The clipboard access service.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The tempstore factory service.
   */
  public function __construct(
    ParagraphsClipboardService $paragraphs_clipboard_service,
    ParagraphsClipboardAccess $paragraphs_clipboard_access,
    EntityRepositoryInterface $entity_repository,
    RequestStack $request_stack,
    PrivateTempStoreFactory $temp_store_factory,
  ) {
    $this->paragraphsClipboardService = $paragraphs_clipboard_service;
    $this->paragraphsClipboardAccess = $paragraphs_clipboard_access;
    $this->entityRepository = $entity_repository;
    $this->requestStack = $request_stack;
    $this->tempStoreFactory = $temp_store_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      LayoutParagraphsAllowedTypesEvent::EVENT_NAME => 'onAllowedTypes',
    ];
  }

  /**
   * Alters the allowed component types.
   *
   * @param \Drupal\layout_paragraphs\Event\LayoutParagraphsAllowedTypesEvent $event
   *   The event object containing the component types.
   */
  public function onAllowedTypes(LayoutParagraphsAllowedTypesEvent $event) {
    $paragraph = $this->paragraphsClipboardService->getParagraphFromClipboard(ParagraphsClipboardService::CLIPBOARD_UUID_KEY);
    if (!$paragraph instanceof Paragraph) {
      return;
    }

    $queryParams = ['sibling_uuid', 'parent_uuid', 'region', 'placement'];
    foreach ($queryParams as $param) {
      $clipboardContent[$param] = $this->requestStack->getCurrentRequest()->query->get($param);
    }

    $layout = $event->getLayout();
    $layoutId = $layout->id();
    $field_definition = $layout->getParagraphsReferenceField()->getFieldDefinition();
    $real_item_count = $layout->getParagraphsReferenceField()->count();
    $cardinality = $field_definition->getFieldStorageDefinition()->getCardinality();

    $access_result = $this->paragraphsClipboardAccess->access($paragraph, $field_definition);
    if ($access_result->isForbidden() || !$this->paragraphsClipboardService->checkCardinality($real_item_count, $cardinality)) {
      return;
    }

    $paragraphLabel = $paragraph->label();

    $this->tempStoreFactory
      ->get('layout_paragraph_paste')
      ->set($layoutId, $clipboardContent);

    $url = Url::fromRoute('layout_paragraphs_clipboard.builder.paste_clipboard_form', [
      'layout_paragraphs_layout' => $layoutId,
    ], [
      'query' => $clipboardContent,
      'attributes' => [
        'class' => ['use-ajax', 'open-insert-form'],
        'data-dialog-options' => Json::encode(['width' => 800]),
        'data-dialog-type' => 'dialog',
        'title' => $paragraphLabel,
      ],
    ]);

    $types = $event->getTypes();

    $types['paste_clipboard'] = [
      'id' => 'paste_clipboard',
      'label' => $this->t('Paste from clipboard'),
      'image' => FALSE,
      'description' => $this->t('Paste a component from the clipboard into the layout.'),
      'is_section' => FALSE,
      'url_object' => Url::fromRoute('layout_paragraphs_clipboard.builder.paste_clipboard', [
        'layout_paragraphs_layout' => $event->getLayout()->id(),
      ], [
        'query' => $event->getContext(),
      ]),
    ];

    $types['paste_clipboard_form'] = [
      'id' => 'paste_clipboard_form',
      'label' => $this->t('Paste from clipboard with edit'),
      'image' => FALSE,
      'description' => $this->t('Paste a component from the clipboard into the layout.'),
      'is_section' => FALSE,
      'url_object' => $url,
    ];

    $event->setTypes($types);
  }

}
