<?php

namespace Drupal\layout_paragraphs_clipboard;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseDialogCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\layout_paragraphs\Utility\Dialog;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Trait to share clipboard-related functionality between controllers.
 */
trait PasteClipboardTrait {


  /**
   * The clipboard service.
   *
   * @var \Drupal\paragraphs_clipboard\ParagraphsClipboardService
   */
  protected $clipboardService;

  /**
   * The clipboard access.
   *
   * @var \Drupal\paragraphs_clipboard\Access\ParagraphsClipboardAccess
   */
  protected $clipboardAccess;

  /**
   * The EntityRepository service.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * Initializes services for the clipboard.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The container interface.
   */
  protected function initClipboardServices(ContainerInterface $container) {
    $this->clipboardService = $container->get('paragraphs_clipboard.paragraphs_clipboard_service');
    $this->clipboardAccess = $container->get('paragraphs_clipboard.paragraphs_clipboard_access');
    $this->entityRepository = $container->get('entity.repository');
  }

  /**
   * Displays an error message and closes the dialog.
   *
   * @param string $message
   *   The error message to display.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  protected function displayErrorMessage(string $message) {
    $response = new AjaxResponse();
    $response->addCommand(new CloseDialogCommand(Dialog::dialogSelector($this->layoutParagraphsLayout)));
    $response->addCommand(new MessageCommand($message, NULL, ['error']));
    if ($this->needsRefresh()) {
      return $this->refreshLayout($response);
    }
    return $response;
  }

  /**
   * Retrieves a paragraph from the clipboard.
   *
   * @param string $key
   *   The clipboard key.
   *
   * @return \Drupal\paragraphs\Entity\Paragraph|null
   *   The clipboard paragraph entity, or NULL if not found.
   */
  protected function getParagraphFromClipboard(string $key) {
    return $this->clipboardService->getParagraphFromClipboard($key);
  }

}
